CREATE OR REPLACE 
PACKAGE MIGRATION_CONVERT AS 
    TYPE NAMES IS TABLE OF VARCHAR2(128);
    
    PROCEDURE CONVERT(projectId IN NUMBER, connId IN NUMBER, srcDb IN VARCHAR2, 
           schemaArr IN MIGRATION_CONVERT.NAMES, tableArr IN MIGRATION_CONVERT.NAMES, capVer IN NUMBER DEFAULT 0);
           
    PROCEDURE CONVERT(projectId IN NUMBER, connId IN NUMBER, srcDb IN VARCHAR2, 
           schemaJArr IN VARCHAR2, tableJArr IN VARCHAR2, capVer IN NUMBER DEFAULT 0);

    PROCEDURE CONVERT(projectId IN NUMBER, connId IN NUMBER, srcDb IN VARCHAR2,
          capVer IN NUMBER DEFAULT 0);

 END MIGRATION_CONVERT;
/


CREATE OR REPLACE
PACKAGE BODY MIGRATION_CONVERT AS
v_captver NUMBER := 0;
v_versionId NUMBER := 0;
missing_ver EXCEPTION;
   
FUNCTION convertArrToStr(namesArr IN NAMES) RETURN CLOB AS
  v_names CLOB := EMPTY_CLOB();
  BEGIN
	  IF namesArr IS NOT NULL THEN
	     FOR i in 1 .. namesArr.COUNT LOOP
	      IF namesArr(i) IS NOT NULL THEN
		      IF i = 1 THEN 
		         v_names := q'[']' || UPPER(namesArr(i)) || q'[']';
		      ELSE
		         v_names := v_names || ', ' || q'[']' || UPPER(namesArr(i)) || q'[']';
		      END IF;
		  END IF;   
	    END LOOP;
   END IF;
   -- DBMS_OUTPUT.PUT_LINE('LIST: ' || v_names); 
   RETURN v_names;
END convertArrToStr;

FUNCTION convertJSONArrStrToNames(jsonArr IN VARCHAR2) RETURN MIGRATION_CONVERT.NAMES AS
  v_namesType MIGRATION_CONVERT.NAMES := MIGRATION_CONVERT.NAMES(NULL);
  BEGIN
	  IF jsonArr IS NOT NULL AND LENGTH(jsonArr) > 0 THEN
		SELECT 
		   VALUE BULK COLLECT into v_namesType
		FROM JSON_TABLE( jsonArr, '$[*]'	COLUMNS (VALUE PATH '$'));
	 END IF;
	 RETURN v_namesType;
  END convertJSONArrStrToNames;

PROCEDURE PreProcessing(nProjectId IN NUMBER, nConnId IN NUMBER, V_SRCDATABASE IN VARCHAR2, 
        v_capver IN NUMBER) AS
 v_captcnt NUMBER;
 v_conv_ver NUMBER;
 cur NUMBER;
 ex NUMBER;
 v_sql CLOB;
 
 BEGIN
	-- v_captver := 
	IF v_capver = 0 or v_capver IS NULL THEN
	   
		   SELECT COUNT(CAPT_VERSION) INTO v_captcnt 
		   FROM  MD_CATALOGS 
		   WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND TYPE = 'CAPTURED';
	   
		   IF v_captcnt = 0 THEN
		      RAISE missing_ver ;
		   ELSIF v_captcnt > 0 THEN
			       SELECT MAX(CAPT_VERSION) INTO v_captver 
				   FROM  MD_CATALOGS 
				   WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND TYPE = 'CAPTURED';
				   IF v_captver = 0 THEN
				     RAISE missing_ver;
				   END IF;	   
		 END IF;
	ELSE
	       SELECT COUNT(CAPT_VERSION) INTO v_captcnt 
		   FROM  MD_CATALOGS 
		   WHERE CAPT_VERSION = v_capver AND PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND TYPE = 'CAPTURED';
		   IF v_captcnt = 0 THEN
		     RAISE missing_ver;
		   END IF;
		   v_captver := v_capver;
	END IF;
	
	
	SELECT MAX(CONV_VERSION) INTO v_conv_ver 
    FROM  MD_CATALOGS 
    WHERE CAPT_VERSION = v_captver AND PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND TYPE = 'CONVERTED';
   
    IF v_conv_ver IS NULL THEN
	    v_conv_ver := 1;
    ELSE
 	    v_conv_ver := v_conv_ver + 1; -- increase the version number by 1.
    END IF;
    v_versionId :=  v_conv_ver;
   DBMS_OUTPUT.PUT_LINE('Converted Version for ' || V_SRCDATABASE || ' is ' || v_versionId);
   
   cur := DBMS_SQL.OPEN_CURSOR;
   v_sql := q'{TRUNCATE TABLE MD_STAGE_CONSTRAINTS}';
   DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
   ex := DBMS_SQL.EXECUTE(cur);
   
   v_sql := q'{TRUNCATE TABLE MD_STAGE_INDEXES}';
   DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
   ex := DBMS_SQL.EXECUTE(cur);
   DBMS_SQL.CLOSE_CURSOR(cur);
   
   COMMIT;
   
EXCEPTION
    WHEN missing_ver THEN
         RAISE missing_ver;
END PreProcessing;


  PROCEDURE ConvertDatabases(nProjectId IN NUMBER, nConnId IN NUMBER, V_SRCDATABASE IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
    v_vId NUMBER;
    BEGIN
	  
   INSERT INTO MD_CATALOGS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION
    )
    (SELECT 
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      'CONVERTED' TYPE,
      v_captver CAPT_VERSION,
      v_versionId CONV_VERSION
    FROM 
      MD_CATALOGS
    WHERE 
      PROJECT_ID  = nProjectId
      AND CONNECTION_ID = nConnId
      AND CAPT_VERSION = v_captver
      AND TYPE          = 'CAPTURED'
    );
  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_CATALOGS.');
  COMMIT;
  END ConvertDatabases;

  PROCEDURE ConvertSchemas(nProjectId IN NUMBER, nConnId IN NUMBER, V_SRCDATABASE IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_schemas CLOB;
  cur NUMBER;
  ex NUMBER;
  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  
  v_sql := q'[
  INSERT
  INTO MD_SCHEMAS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      SCHEMA_NAME,
      ORACLE_NAME,
      USER_NAMESPACE
    )
    (
      SELECT
        PROJECT_ID,
        CONNECTION_ID,
        CATALOG_NAME,
        TYPE,
        :captvId CAPT_VERSION,
        :verId CONV_VERSION, 
        SCHEMA_NAME,
        DECODE ( (ROW_NUMBER() OVER (PARTITION BY UPPER(SCHTAB.ORACLE_NAME), SCHTAB.USER_NAMESPACE ORDER BY SCHTAB.ORACLE_NAME)), 
                  1, SCHTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
                  MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(SCHTAB.ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(SCHTAB.ORACLE_NAME), USER_NAMESPACE ORDER BY SCHTAB.ORACLE_NAME)), 128)) ORACLE_NAME,
        
        USER_NAMESPACE	            
      FROM 
        (SELECT 
            PROJECT_ID,
            CONNECTION_ID,
            CATALOG_NAME,
            'CONVERTED' TYPE,
            SCHEMA_NAME,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SCHEMA_NAME) ORACLE_NAME,
            USER_NAMESPACE 
          FROM 
            MD_SCHEMAS
          WHERE 
            PROJECT_ID  = :projectId
            AND CONNECTION_ID = :svrId
            AND CAPT_VERSION = :captvId
            AND TYPE = 'CAPTURED'
   ]';
   
   cur := DBMS_SQL.OPEN_CURSOR;
   IF DBMS_LOB.GETLENGTH(v_schemas) = 0 THEN
	  v_sql := v_sql;
   ELSE
     v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
     v_sql := v_sql || v_schemas;
   END IF;
   
   v_sql := v_sql || ')SCHTAB)';
	  
   -- DBMS_OUTPUT.PUT_LINE(v_sql);
   DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
   DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
   DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
   DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
   DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
   ex := DBMS_SQL.EXECUTE(cur);
   DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_SCHEMAS.');
   DBMS_SQL.CLOSE_CURSOR(cur);
   COMMIT;
  END ConvertSchemas;

  PROCEDURE ConvertTables(nProjectId IN NUMBER, nConnId IN NUMBER, V_SRCDATABASE IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);	  
  v_sql := q'[
    INSERT
	  INTO MD_TABLES
	    (
	      PROJECT_ID,
	      CONNECTION_ID,
	      CATALOG_NAME,
	      TYPE,
          CAPT_VERSION, 
          CONV_VERSION,
          SCHEMA_NAME,
	      TABLE_NAME,
	      ORACLE_NAME,
	      ISINDEXTABLE,
	      QUALIFIED_NATIVE_NAME,
	      SCHEMA_NAMESPACE,
	      COMMENTS,
	      NUM_ROWS
	    )
	    (SELECT
            SRC.PROJECT_ID,
	        SRC.CONNECTION_ID,
	        SRC.CATALOG_NAME,
	        SRC.TYPE,
            :captvId CAPT_VERSION, 
            :verId CONV_VERSION,
            SRC.SCHEMA_NAME,
	        SRC.TABLE_NAME,
	        TAB.ORACLE_NAME,
	        SRC.ISINDEXTABLE,
	        SRC.QUALIFIED_NATIVE_NAME,
	        SRC.SCHEMA_NAMESPACE,
	        SRC.COMMENTS,
	        SRC.NUM_ROWS
	      FROM
	        (SELECT 
	          ROWID RID,
	          PROJECT_ID,
	          CONNECTION_ID,
	          CATALOG_NAME,
	          'CONVERTED' TYPE,
	          SCHEMA_NAME,
	          TABLE_NAME,
	          ISINDEXTABLE,
	          QUALIFIED_NATIVE_NAME,
	          SCHEMA_NAMESPACE,
	          ORACLE_NAME,
	          COMMENTS,
	          NUM_ROWS
	        FROM 
	          MD_TABLES
	        WHERE 
	          PROJECT_ID  = :projectId
	          AND CONNECTION_ID = :svrId
	          AND TYPE          = 'CAPTURED' 
              AND CAPT_VERSION = :captvId]';
   
	v_sql1 := q'[
            ) SRC ,
	        (SELECT 
	          DECODE ( (ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 
	             1, UTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
	             MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(UTAB.ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 128) ) ORACLE_NAME,
	          UTAB.RID RID
	        FROM
	          (SELECT 
	            ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	           FROM 
	            MD_TABLES
	           WHERE 
	            SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	            AND TYPE = 'CAPTURED' AND CAPT_VERSION = :captvId
	          UNION
	          SELECT 
	            ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(VIEW_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM 
	            MD_VIEWS
	          WHERE 
	            SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID 
	            AND TYPE = 'CAPTURED' AND CAPT_VERSION = :captvId
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SEQUENCE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_SEQUENCES
	          WHERE 
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID 
	          AND TYPE             = 'CAPTURED' AND CAPT_VERSION = :captvId
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PROGRAM_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_STORED_PROGRAMS
	          WHERE 
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE = 'CAPTURED' AND CAPT_VERSION = :captvId
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PACKAGE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_PACKAGES
	          WHERE
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE             = 'CAPTURED' AND CAPT_VERSION = :captvId
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(DATA_TYPE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_USER_DEFINED_DATA_TYPES
	          WHERE 
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE = 'CAPTURED' AND CAPT_VERSION = :captvId
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SYNONYM_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_SYNONYMS
	          WHERE 
	          PRIVATE_VISIBILITY = 'Y'
	          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE = 'CAPTURED' AND CAPT_VERSION = :captvId
	          ) UTAB
	        ) TAB
	      WHERE SRC.RID = TAB.RID
	    )]';
	  
	cur := DBMS_SQL.OPEN_CURSOR;
	
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas || v_sql1;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables || v_sql1;
	ELSE
	  v_sql := v_sql || v_sql1;
	END IF;
    
	-- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
    DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_TABLES.');
    DBMS_SQL.CLOSE_CURSOR(cur);
	  
    COMMIT;
  END ConvertTables;

  PROCEDURE ConvertColumns(nProjectId IN NUMBER, nConnId IN NUMBER, V_SRCDATABASE IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);	  
  v_sql := q'{
  INSERT
  INTO MD_COLUMNS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION, 
      CONV_VERSION,
      SCHEMA_NAME,
      TABLE_NAME,
      COLUMN_NAME,
      ORACLE_NAME,
      COLUMN_ORDER,
      COLUMN_TYPE,
      PRECISION,
      SCALE,
      NULLABLE,
      DEFAULT_VALUE,
      DATATYPE_TRANSFORMED_FLAG,
      ISAUTOINC,
      ISNEWID,
      AUTOINC_SEEDVALUE,
      AUTOINC_INCREMENTVALUE,
      AUTOINC_LASTVALUE,
      COLUMN_NAMESPACE,
      COMMENTS
    )
    (SELECT
        COL.PROJECT_ID,
        COL.CONNECTION_ID,
        COL.CATALOG_NAME,
        'CONVERTED' TYPE,
        :captvId CAPT_VERSION, 
        :verId CONV_VERSION,
        COL.SCHEMA_NAME,
        COL.TABLE_NAME TABLE_NAME,
        COL.COLUMN_NAME,
        DECODE ( (ROW_NUMBER() OVER (PARTITION BY TABLE_NAME, UPPER(COL.ORACLE_NAME), COL.COLUMN_NAMESPACE ORDER BY COLUMN_ORDER)), 
            1, COL.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
            MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(COL.ORACLE_NAME), '_'  || TO_CHAR(COLUMN_ORDER), 128) ) ORACLE_NAME,
        COL.COLUMN_ORDER,
        CASE  WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'CHAR' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000)
                  THEN 'CLOB'
              WHEN  (DTRULE.TARGET_DATA_TYPE_NAME = 'VARCHAR2' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 4000)
                  THEN 'CLOB'
              WHEN  (DTRULE.TARGET_DATA_TYPE_NAME = 'NVARCHAR2' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000)
                  THEN 'CLOB'
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'TIMESTAMP')
                  THEN 'TIMESTAMP'
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'RAW' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000) 
                  THEN 'BLOB'
              ELSE DTRULE.TARGET_DATA_TYPE_NAME END COLUMN_TYPE,
        CASE  WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'CHAR' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000)
                  THEN NULL
              WHEN  (DTRULE.TARGET_DATA_TYPE_NAME = 'VARCHAR2' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 4000)
                  THEN NULL
              WHEN  (DTRULE.TARGET_DATA_TYPE_NAME = 'NVARCHAR2' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000)
                  THEN NULL
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'TIMESTAMP')
                  THEN 9
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'RAW' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000) 
                  THEN NULL
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'NUMBER' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 38) 
                  THEN 38    
              ELSE DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) END PRECISION,
        CASE  WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'CHAR' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000)
                  THEN NULL
              WHEN  (DTRULE.TARGET_DATA_TYPE_NAME = 'VARCHAR2' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 4000)
                  THEN NULL
              WHEN  (DTRULE.TARGET_DATA_TYPE_NAME = 'NVARCHAR2' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000)
                  THEN NULL
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'TIMESTAMP')
                  THEN NULL
              WHEN (DTRULE.TARGET_DATA_TYPE_NAME = 'RAW' AND DECODE(COL.PRECISION, NULL, DTRULE.TARGET_PRECISION, COL.PRECISION) > 2000) 
                  THEN NULL
              ELSE DECODE(COL.SCALE, NULL, DTRULE.TARGET_SCALE, COL.SCALE) END SCALE,
        COL.NULLABLE,
        COL.DEFAULT_VALUE,         
        'Y' DATATYPE_TRANSFORMED_FLAG,
        COL.ISAUTOINC,
        COL.ISNEWID,
        COL.AUTOINC_SEEDVALUE,
        COL.AUTOINC_INCREMENTVALUE,
        COL.AUTOINC_LASTVALUE,
        COL.COLUMN_NAMESPACE,
        COL.COMMENTS
      FROM
        (SELECT ROWID RID,
          PROJECT_ID,
          CONNECTION_ID,
          CATALOG_NAME,
          TYPE,
          CAPT_VERSION,
          SCHEMA_NAME,
          TABLE_NAME,
          COLUMN_NAME,
          MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(COLUMN_NAME) ORACLE_NAME ,
          COLUMN_ORDER,
          COLUMN_TYPE,
          PRECISION,
          SCALE,
          NULLABLE,
          CASE WHEN :v_srcdatabase = 'SYBASE' 
                THEN CASE  
    				  WHEN REGEXP_LIKE(DEFAULT_VALUE,  '\(?(\w+)\)?', 'i') AND ( COLUMN_TYPE = 'binary' OR COLUMN_TYPE = 'varbinary' )
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(\w+)\)?', q'[UTL_RAW.CAST_TO_RAW('\1')]', 1, 1, 'i') 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^create\s+\w+\s+as+\s+getDate\$?\(?\)?', 'i') AND COLUMN_TYPE = 'datetime'
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, '^create\s+\w+\s+as+\s+getDate\$?\(?\)?', 'SYSDATE', 1, 1, 'i'),CHR(10),''), CHR(13),''))
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^create\s+\w+\s+as+\s+(.*)', 'i') AND COLUMN_TYPE = 'char'
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, '^create\s+\w+\s+as+\s+(.*)', q'['\1']', 1, 1, 'i'),CHR(10),''), CHR(13),''))
                      WHEN INSTR(UPPER(DEFAULT_VALUE),'CONVERT', 1, 1) = 1
                        THEN 'SYSDATE'
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '(getDate)\$?\(?\)?', 'i') -- SYSDATE
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, '(getDate)\$?\(?\)?', 'SYSDATE', '1', 1, 'i'),CHR(10),''), CHR(13),''))
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\((\d{1})\)', 'i') -- (1) 
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, '\((\d{1})\)',  '\1', 1, 1, 'i'),CHR(10),''), CHR(13),''))
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[\(\'(\w{1})\'\)]', 'i') -- ('N') 
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, q'[\(\'(\w{1})\'\)]',  q'['\1']', 1, 1, 'i'),CHR(10),''), CHR(13),''))
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^(\"|\'|)GETUTCDATE\(\)(\"|\'|)$]', 'i') -- GETUTCDATE 
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, q'[^(\"|\'|)GETUTCDATE\(\)(\"|\'|)$]', 'SYS_EXTRACT_UTC(SYSTIMESTAMP)', 1, 1, 'i'),CHR(10),''), CHR(13),''))
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(''\)') -- Empty String with braces
                        THEN TRIM(REPLACE(REPLACE( REGEXP_REPLACE(DEFAULT_VALUE, '\(''\)', ' '),CHR(10),''), CHR(13),''))
                     -- ELSE REGEXP_REPLACE(DEFAULT_VALUE,'(^[[:space:]]*|[[:cntrl:]]*$)') 
                      ELSE TRIM(REPLACE(REPLACE(DEFAULT_VALUE,CHR(10),''), CHR(13),''))
                 END
               WHEN :v_srcdatabase = 'DB2'
                  THEN CASE
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, 'Date\$?\(\)', 'i') -- DATE 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, 'Date\$?\(\)', 'SYSDATE', 1, 1, 'i')
                      	WHEN REGEXP_LIKE(DEFAULT_VALUE, 'Now\$?\(\)', 'i') -- NOW 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, 'Now\$?\(\)', 'SYSDATE', 1, 1, 'i')
                      	WHEN REGEXP_LIKE(DEFAULT_VALUE, 'Time\$?\(\)', 'i') -- TIME 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, 'Time\$?\(\)', 'SYSDATE', 1, 1, 'i') 
                      	WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^\"SYSIBM\"\.\"(\w+)\"(\(\'.*\'\))$]', 'i') AND COLUMN_TYPE = 'BLOB'  -- convert BLOB default values
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[^\"SYSIBM\"\.\"(\w+)\"(\(\'.*\'\))$]', q'[UTL_RAW.CAST_TO_RAW\2]', 1, 1, 'i')  
                      	WHEN REGEXP_LIKE(DEFAULT_VALUE, '^(current\s+date|current_date|Date)\$?\(?\)?', 'i') -- DATE 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '^(current\s+date|current_date|Date)\$?\(?\)?', 'SYSDATE', 1, 1, 'i')
                      	WHEN REGEXP_LIKE(DEFAULT_VALUE, '(current)\s+(timestamp)(\(?\d?\)?)', 'i') 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '(current)\s+(timestamp)(\(?\d?\)?)', '\1_\2\3', 1, 1, 'i')
                      	WHEN REGEXP_LIKE(DEFAULT_VALUE, '(current)\s+(time)(\(?\d?\)?)', 'i') 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '(current)\s+(time)(\(?\d?\)?)', '\1_\2STAMP\3', 1, 1, 'i')        
                    --    WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(^\'\'$)]')
                    --      THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(^\'\'$)]', ' ') -- Empty String Rule (Oracle treats empty string as NULL)
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^\"SYSIBM\"\.\"(\w+)\"(\(\'.*\'\))$]', 'i') -- Remove SYSIBM
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[^\"SYSIBM\"\.\"(\w+)\"(\(\'.*\'\))$]', q'[\1\2]', 1, 1, 'i')
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, '^(\")*(.*)(\")$') -- Remove Double Quotes
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '^(\")*(.*)(\")$', '\2',  1, 1, 'i')
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, '^No$', 'i') -- Boolean No Rule 1
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '^No$', 0,  1, 1, 'i')
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, '^False$', 'i') -- Boolean No Rule 2
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '^False$', 0,  1, 1, 'i')
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, '^Yes$', 'i') -- Boolean Yes Rule 1
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '^Yes$', -1,  1, 1, 'i')
                        WHEN REGEXP_LIKE(DEFAULT_VALUE, '^True$', 'i') -- Boolean Yes Rule 2
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, '^True$', -1,  1, 1, 'i')
                        ELSE DEFAULT_VALUE END 
               WHEN :v_srcdatabase = 'SQLSERVER'
                 THEN CASE 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE,  '\(?(\w+)\)?', 'i') AND ( COLUMN_TYPE = 'binary' OR COLUMN_TYPE = 'varbinary' )
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(\w+)\)?', q'[UTL_RAW.CAST_TO_RAW('\1')]', 1, 1, 'i') 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(?(getDate)\$?\(?\)?\)?', 'i') -- SYSDATE
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(getDate)\$?\(?\)?\)?', 'SYSDATE', '1', 1, 'i') 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(?(host_name)\$?\(?\)?\)?', 'i') -- SYSDATE
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(host_name)\$?\(?\)?\)?', q'[SYS_CONTEXT('USERENV','HOST')]', '1', 1, 'i')  
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(?(user_name)\$?\(?\)?\)?', 'i') -- SYSDATE
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(user_name)\$?\(?\)?\)?', q'[SYS_CONTEXT('USERENV','SESSION_USER')]', '1', 1, 'i')   
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(?(app_name)\$?\(?\)?\)?', 'i') -- SYSDATE
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(app_name)\$?\(?\)?\)?', q'[USERENV('sessionid')]', '1', 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(?(newid)\$?\(?\)?\)?', 'i') -- newid()
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(?(newid)\$?\(?\)?\)?', 'sys_guid()', '1', 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^(\"|\'|)GETUTCDATE\(\)(\"|\'|)$]', 'i') -- Sybase GETUTCDATE 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[^(\"|\'|)GETUTCDATE\(\)(\"|\'|)$]', 'SYS_EXTRACT_UTC(SYSTIMESTAMP)', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^\"user_name \"$', 'i') -- Sybase GETUTCDATE 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[^(\"|\'|)GETUTCDATE\(\)(\"|\'|)$]', 'SYS_EXTRACT_UTC(SYSTIMESTAMP)', 1, 1, 'i')  
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '\(''\)') -- Empty String with Braces
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '\(''\)', ' ') 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^(\()(.*)(\))$', 'i') -- Remove Braces 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^(\()(.*)(\))$', '\2', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^''$') -- Empty String without Braces
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^''$', ' ')
                      ELSE DEFAULT_VALUE END
               WHEN :v_srcdatabase = 'TERADATA'
                 THEN CASE
                 --   WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'\w+\')]', 'i') 
                 --       THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'\w+\')]', q'[\1]'||' #1', 1, 1, 'i')    
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^(current\s+date|current_date|Date)\$?\(?\)?', 'i') -- DATE 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^(current\s+date|current_date|Date)\$?\(?\)?', 'SYSDATE', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, 'Now\$?\(\)', 'i') -- NOW 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, 'Now\$?\(\)', 'SYSDATE', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^(\"|\'|)Time(\"|\'|)$]', 'i') AND COLUMN_TYPE = 'REAL'-- GETDATE
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[^(\"|\'|)Time(\"|\'|)$]', q'[TO_CHAR(current_timestamp, 'hhmiss.ff2')]', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^(CURRENT_TIMESTAMP)(\(?\d?\)?)]', 'i') 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[^(CURRENT_TIMESTAMP)(\(?\d?\)?)]', '\1\2', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^(Current\s+Time)\(?\d?\)?', 'i') and (COLUMN_TYPE = 'ANSITIMEWITHTIMEZONE' OR COLUMN_TYPE = 'ANSITIME')
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^(Current\s+Time)\(?\d?\)?', 'SYSDATE', 1, 1, 'i')   
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '(current)\s+(timestamp)(\(?\d?\)?)', 'i') 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '(current)\s+(timestamp)(\(?\d?\)?)', '\1_\2\3', 1, 1, 'i') 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\d{4}-((((0[13578])|(1[02]))-(([0-2][0-9])|(3[01])))|(((0[469])|(11))-(([0-2][0-9])|(30)))|(02-[0-2][0-9])))]', 'i') AND COLUMN_TYPE = 'DATE'
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\d{4}-((((0[13578])|(1[02]))-(([0-2][0-9])|(3[01])))|(((0[469])|(11))-(([0-2][0-9])|(30)))|(02-[0-2][0-9])))]', q'[DATE '\1']',1, 1)                         
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[^[0-9]{1}[0-9]{3}-(0[1-9]{1}|1[0-2]{1})-([0-2]{1}[1-9]{1}|3[0-1]{1})\s+([0-1]{1}[0-9]{1}|2[0-3]{1}):[0-5]{1}[0-9]{1}:[0-5]{1}[0-9]{1}$]') AND COLUMN_TYPE = 'TIMESTAMPWITHOUTTIMEZONE'
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(^[0-9]{1}[0-9]{3}-(0[1-9]{1}|1[0-2]{1})-([0-2]{1}[1-9]{1}|3[0-1]{1})\s+([0-1]{1}[0-9]{1}|2[0-3]{1}):[0-5]{1}[0-9]{1}:[0-5]{1}[0-9]{1}$)]',q'[TIMESTAMP '\1']',1, 1)              
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, 'null', 'i') 
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, 'null', 'NULL', 1, 1, 'i')  
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'?(\w+\s)(\w+)?\'?)]', 'i')   AND COLUMN_TYPE = 'CHARFIXEDLATIN'
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'?(\w+\s)(\w+)?\'?)]',  '\1', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'?(\w+\s*)\'?)]', 'i')   AND (COLUMN_TYPE = 'CHARFIXEDLATIN' OR COLUMN_TYPE = 'CHARVARYINGLATIN')
                          THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'?(\w+\s*)\'?)]',  q'['\2']', 1, 1, 'i') 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'?\w+\s+\'?)]', 'i')
                         THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'?\w+\s+\'?)]', q'[\1]', 1, 1, 'i')   
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'\w+\')(XBV|XB|V|F)]', 'i') 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'\w+\')(XBV|XB|V|F)]', '\1', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'\')(XBV|XB|V|F)]', 'i') AND COLUMN_TYPE = 'BYTEVARYING'
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'\')(XBV|XB|V|F)]', 'NULL', 1, 1, 'i')  
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[\'\']', 'i') 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[\'\']', 'NULL', 1, 1, 'i')  
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, 'UNKNOWN', 'i') 
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, 'UNKNOWN', 'UNKNOWN', 1, 1, 'i')
                    --  WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\d{5}\-\d{4})?]', 'i')  
                     --   THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\d{5}\-\d{4})?]', '\1'||' *15', 1, 1, 'i')                    
                   --    WHEN REGEXP_LIKE(DEFAULT_VALUE, '(0)\.', 'i')
                    --    THEN REGEXP_REPLACE(DEFAULT_VALUE, '(0)\.', '\1'||' *16', 1, 1, 'i')   
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '(\w+\.{1}\w+)', 'i')  AND COLUMN_TYPE = 'REAL'
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '(\w+\.{1}\w+)', q'[\1]', 1, 1, 'i')
                   --   WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(\'\w{1}\')]', 'i')
                    --    THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(\'\w{1}\')]',  q'[\1]'||' *18', 1, 1, 'i')    
                    --  WHEN REGEXP_LIKE(DEFAULT_VALUE, '(\w{1})', 'i')
                    --    THEN REGEXP_REPLACE(DEFAULT_VALUE, '(\w{1})',  q'['\1']'||' *19', 1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^(\")*(.*)(\")$') -- Remove Double Quotes
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^(\")*(.*)(\")$', '\2',  1, 1, 'i')
                       WHEN REGEXP_LIKE(DEFAULT_VALUE, '^No$', 'i') -- Boolean No Rule 1
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^No$', 0,  1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^False$', 'i') -- Boolean No Rule 2
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^False$', 0,  1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^Yes$', 'i') -- Boolean Yes Rule 1
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^Yes$', -1,  1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^True$', 'i') -- Boolean Yes Rule 2
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^True$', -1,  1, 1, 'i')
                      ELSE DEFAULT_VALUE END 
               WHEN :v_srcdatabase = 'POSTGRESQL' or :v_srcdatabase = 'AWS_REDSHIFT'
                 THEN CASE 
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(*)::bpchar]', 'i') -- CHARACTER
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(*)::bpchar]', '', '1', 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(*)::character varying]', 'i') -- VARCHAR2
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, q'[(*)::character varying]', '', '1', 1, 'i') 
                      WHEN DEFAULT_VALUE = q'[('now'::text)::timestamp without time zone]' ---- SYSDATE
                          THEN replace(REGEXP_REPLACE(DEFAULT_VALUE, q'[(*)::timestamp without time zone]', '', '1', 1, 'i'),q'[('now'::text)]', 'SYSDATE')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(*)::date]', 'i') -- DATE
                        THEN q'[(to_date(]' || REPLACE(DEFAULT_VALUE,'::date', '') ||q'[, 'yyyy-mm-dd'))]'              
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(*)::timestamp without time zone]', 'i') -- timestamp without time zone
                        THEN q'[(to_timestamp(]' || REPLACE(DEFAULT_VALUE,'::timestamp without time zone', '') ||q'[, 'YYYY-MM-DD HH24:MI:SS'))]'
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, q'[(*)::timestamp with time zone]', 'i') -- timestamp with time zone
                        THEN q'[(to_timestamp_tz(]' || REPLACE(DEFAULT_VALUE,'::timestamp with time zone', '') ||q'[, 'YYYY-MM-DD HH24:MI:SSTZH:TZM'))]'  
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^False$', 'i') -- Boolean No Rule 2
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^False$', q'['N']',  1, 1, 'i')
                      WHEN REGEXP_LIKE(DEFAULT_VALUE, '^True$', 'i') -- Boolean Yes Rule 2
                        THEN REGEXP_REPLACE(DEFAULT_VALUE, '^True$', q'['Y']',  1, 1, 'i')
                      ELSE DEFAULT_VALUE END
               ELSE DEFAULT_VALUE 
          END DEFAULT_VALUE,
          DATATYPE_TRANSFORMED_FLAG,
          ISAUTOINC,
          ISNEWID,
          AUTOINC_SEEDVALUE,
          AUTOINC_INCREMENTVALUE,
          AUTOINC_LASTVALUE,
          COLUMN_NAMESPACE,
          COMMENTS
        FROM MD_COLUMNS
        WHERE PROJECT_ID  = :projectId
        AND CONNECTION_ID = :svrId
        AND TYPE          = 'CAPTURED'
        AND CAPT_VERSION = :captvId
        ) COL,
        ( SELECT * FROM MD_DATATYPE_TRANSFORM_RULE WHERE PROJECT_ID = :projectId AND CONNECTION_ID = :svrId
        ) DTRULE
      WHERE UPPER(COL.COLUMN_TYPE)          = UPPER(DTRULE.SOURCE_DATA_TYPE_NAME)
      AND ( DTRULE.SOURCE_PRECISION IS NULL
      OR ( DTRULE.SOURCE_PRECISION   = COL.PRECISION
      AND ( DTRULE.SOURCE_SCALE     IS NULL
      OR DTRULE.SOURCE_SCALE         = COL.SCALE ) ) )
      -- from sqlserver convert
      -- i.e. varchar(MAX ==-999) only matches varchar(MAX) rule -> resonable conclusing - max must have its own data type mapping which can be to blob/clob/nclob 
      -- or even varchar2(32767) or varchar2(4000) if the data allows.
      AND (COL.PRECISION != -999 OR COL.PRECISION IS NULL OR COL.PRECISION = -999 AND DTRULE.SOURCE_PRECISION = -999) 
      AND COL.PROJECT_ID             = :projectId
      AND COL.CONNECTION_ID          = :svrId
      AND COL.CAPT_VERSION = :captvId
      AND COL.TYPE                   = 'CAPTURED'
    }';
   
    cur := DBMS_SQL.OPEN_CURSOR;
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;
    
    v_sql := v_sql || ')';
	-- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
    DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
    DBMS_SQL.BIND_VARIABLE(cur, 'v_srcdatabase', V_SRCDATABASE);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_COLUMNS.');
    DBMS_SQL.CLOSE_CURSOR(cur);
   COMMIT;
  END ConvertColumns;

  PROCEDURE ConstraintCleanup(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2) AS
  BEGIN
    
    INSERT
	INTO MD_STAGE_CONSTRAINTS
	(
	  PROJECT_ID,
	  CONNECTION_ID,
	  CATALOG_NAME,
	  TYPE,
      CAPT_VERSION,
      CONV_VERSION,
	  SCHEMA_NAME,
	  TABLE_NAME,
	  CONSTRAINT_NAME,
	  COLUMN_NAME,
	  ORACLE_NAME,
	  DELETE_CLAUSE,
	  CONSTRAINT_TYPE,
	  DETAIL_ORDER,
	  REFCATALOG_NAME,
	  REFSCHEMA_NAME,
	  REFTABLE_NAME,
	  REFCOLUMN_NAME,
	  CONSTRAINT_TEXT,
	  ISRULE,
	  ISCLUSTEREDINDEX,
	  LANGUAGE,
	  COMMENTS,
	  CONSTRAINT_NAMESPACE,
	  SECURITY_GROUP_ID, 
	  CREATED_ON
    
--	  CREATED_BY, 
--	  LAST_UPDATED_ON, 
--	  LAST_UPDATED_BY
	)
	(
		    SELECT
  			  PROJECT_ID, 
              CONNECTION_ID, 
              CATALOG_NAME,
              TYPE,
              CAPT_VERSION,
              CONV_VERSION,
              SCHEMA_NAME,
              TABLE_NAME,
              CONSTRAINT_NAME,
              COLUMN_NAME,
              ORACLE_NAME,
              DELETE_CLAUSE,
              CONSTRAINT_TYPE, 
              DETAIL_ORDER,
              REFCATALOG_NAME,
              REFSCHEMA_NAME,
              REFTABLE_NAME, 
              REFCOLUMN_NAME,
              CONSTRAINT_TEXT,
              ISRULE, 
              ISCLUSTEREDINDEX, 
              LANGUAGE,
              COMMENTS,
              CONSTRAINT_NAMESPACE,
              SECURITY_GROUP_ID, 
              CREATED_ON
            FROM 
              MD_CONSTRAINTS 
            WHERE 
              TYPE = 'CAPTURED'
              AND CONSTRAINT_TYPE = 'CHECK'
              -- AND SCHEMA_NAME = SCHEMA_NAME
              -- AND TABLE_NAME = TABLE_NAME
			  AND PROJECT_ID  = nProjectId
			  AND CONNECTION_ID = nConnId
              AND CAPT_VERSION = v_captver
    );
	COMMIT;
	
	
	INSERT
	INTO MD_STAGE_CONSTRAINTS
	(
	  PROJECT_ID,
	  CONNECTION_ID,
	  CATALOG_NAME,
	  TYPE,
      CAPT_VERSION,
      CONV_VERSION,
	  SCHEMA_NAME,
	  TABLE_NAME,
	  CONSTRAINT_NAME,
	  COLUMN_NAME,
	  ORACLE_NAME,
	  DELETE_CLAUSE,
	  CONSTRAINT_TYPE,
	  DETAIL_ORDER,
	  REFCATALOG_NAME,
	  REFSCHEMA_NAME,
	  REFTABLE_NAME,
	  REFCOLUMN_NAME,
	  ISRULE,
	  ISCLUSTEREDINDEX,
	  LANGUAGE,
	  COMMENTS,
	  CONSTRAINT_NAMESPACE,
	  SECURITY_GROUP_ID, 
	  CREATED_ON
    
--	  CREATED_BY, 
--	  LAST_UPDATED_ON, 
--	  LAST_UPDATED_BY
	)
	(
		      
			SELECT
        cst.PROJECT_ID, 
        cst.CONNECTION_ID, 
        cst.CATALOG_NAME,
        cst.TYPE,
        cst.CAPT_VERSION,
        cst.CONV_VERSION,
        cst.SCHEMA_NAME,
        cst.TABLE_NAME,
        cst.CONSTRAINT_NAME,
        cst.COLUMN_NAME,
        cst.ORACLE_NAME,
        cst.DELETE_CLAUSE,
        cst.CONSTRAINT_TYPE, 
        cst.DETAIL_ORDER,
        cst.REFCATALOG_NAME,
        cst.REFSCHEMA_NAME,
        cst.REFTABLE_NAME, 
        cst.REFCOLUMN_NAME,
        cst.ISRULE, 
        cst.ISCLUSTEREDINDEX, 
        cst.LANGUAGE,
        cst.COMMENTS,
        cst.CONSTRAINT_NAMESPACE,
        cst.SECURITY_GROUP_ID, 
        cst.CREATED_ON
      FROM 
        MD_CONSTRAINTS cst, MD_COLUMNS col 
      WHERE 
        cst.TYPE = 'CAPTURED'
        AND cst.CONSTRAINT_TYPE = 'PK'
        -- AND SCHEMA_NAME = SCHEMA_NAME
        -- AND TABLE_NAME = TABLE_NAME
		AND cst.PROJECT_ID  = nProjectId
		AND cst.CONNECTION_ID = nConnId
        AND cst.CAPT_VERSION = v_captver
		AND cst.PROJECT_ID  = col.PROJECT_ID
		AND cst.CONNECTION_ID = col.CONNECTION_ID
		AND col.TYPE = 'CONVERTED'
		AND cst.SCHEMA_NAME = col.SCHEMA_NAME
    AND cst.TABLE_NAME = col.TABLE_NAME
    AND cst.COLUMN_NAME = col.COLUMN_NAME
		AND col.COLUMN_TYPE NOT IN ( 'BLOB', 'CLOB','RAW','NCLOB','LONG','LONG RAW','BFILE','TIMESTAMP WITH TIME ZONE')
	
	  UNION
			(
			SELECT
        PROJECT_ID, 
        CONNECTION_ID, 
        CATALOG_NAME,
        TYPE,
        CAPT_VERSION,
        CONV_VERSION, 
        SCHEMA_NAME,
        TABLE_NAME,
        CONSTRAINT_NAME,
        COLUMN_NAME,
        ORACLE_NAME,
        DELETE_CLAUSE,
        CONSTRAINT_TYPE, 
        DETAIL_ORDER,
        REFCATALOG_NAME,
        REFSCHEMA_NAME,
        REFTABLE_NAME, 
        REFCOLUMN_NAME,
        ISRULE, 
        ISCLUSTEREDINDEX, 
        LANGUAGE,
        COMMENTS,
        CONSTRAINT_NAMESPACE,
        SECURITY_GROUP_ID, 
        CREATED_ON
      FROM 
        MD_CONSTRAINTS 
      WHERE 
        TYPE = 'CAPTURED'
        AND CONSTRAINT_TYPE = 'UNIQUE'
        -- AND SCHEMA_NAME = SCHEMA_NAME
        -- AND TABLE_NAME = TABLE_NAME
        AND PROJECT_ID  = nProjectId
        AND CONNECTION_ID = nConnId
        AND CAPT_VERSION = v_captver 
			MINUS
			/*
			Remove unwanted UNIQUE keys
			Composite keys needs to be resolved.
			Bug 16998890 - MIGRATION GENERATION : GENERATION OF UNWANTED UNIQUE KEYS 
			Remove all UniqueKeys that are created on the same table columns that has Primary keys created. 
			4.0 MIGRATION Package : PROCEDURE remove_unwanted_uniquekeys(p_connectionid MD_CONNECTIONS.ID%TYPE)
			*/ 
      SELECT
        PROJECT_ID, 
        CONNECTION_ID, 
        CATALOG_NAME,
        TYPE,
        CAPT_VERSION,
        CONV_VERSION,
        SCHEMA_NAME,
        TABLE_NAME,
        CONSTRAINT_NAME,
        COLUMN_NAME,
        ORACLE_NAME,
        DELETE_CLAUSE,
        CONSTRAINT_TYPE, 
        DETAIL_ORDER,
        REFCATALOG_NAME,
        REFSCHEMA_NAME,
        REFTABLE_NAME, 
        REFCOLUMN_NAME,
        ISRULE, 
        ISCLUSTEREDINDEX, 
        LANGUAGE,
        COMMENTS,
        CONSTRAINT_NAMESPACE,
        SECURITY_GROUP_ID, 
        CREATED_ON
      FROM 
         ( 
          SELECT
            MC2.PROJECT_ID, 
            MC2.CONNECTION_ID, 
            MC2.CATALOG_NAME,
            MC2.TYPE,
            MC2.CAPT_VERSION,
            MC2.CONV_VERSION,
            MC2.SCHEMA_NAME,
            MC2.TABLE_NAME,
            MC2.CONSTRAINT_NAME,
            MC2.COLUMN_NAME,
            MC2.ORACLE_NAME,
            MC2.DELETE_CLAUSE,
            MC2.CONSTRAINT_TYPE, 
            MC2.DETAIL_ORDER,
            MC2.REFCATALOG_NAME,
            MC2.REFSCHEMA_NAME,
            MC2.REFTABLE_NAME, 
            MC2.REFCOLUMN_NAME,
            MC2.ISRULE, 
            MC2.ISCLUSTEREDINDEX, 
            MC2.LANGUAGE,
            MC2.COMMENTS,
            MC2.CONSTRAINT_NAMESPACE,
            MC2.SECURITY_GROUP_ID, 
            MC2.CREATED_ON
          FROM 
            MD_CONSTRAINTS MC1,
            MD_CONSTRAINTS MC2
          WHERE 
            MC1.TYPE = 'CAPTURED'
            AND MC1.PROJECT_ID  = nProjectId
            AND MC1.CONNECTION_ID = nConnId
            AND MC1.CAPT_VERSION = v_captver
            AND MC1.CONSTRAINT_TYPE = 'PK' 
            AND MC2.CONSTRAINT_TYPE = 'UNIQUE' 
            AND MC1.PROJECT_ID = MC2.PROJECT_ID
            AND MC1.CONNECTION_ID = MC2.CONNECTION_ID
            AND MC1.TYPE = MC2.TYPE
            AND MC1.SCHEMA_NAME = MC2.SCHEMA_NAME
            AND MC1.TABLE_NAME = MC2.TABLE_NAME
            AND MC1.COLUMN_NAME = MC2.COLUMN_NAME
            AND MC1.CONSTRAINT_NAME <> MC2.CONSTRAINT_NAME
          )
			)  
      UNION  
			(
        SELECT
          PROJECT_ID, 
          CONNECTION_ID, 
          CATALOG_NAME,
          TYPE,
          CAPT_VERSION,
          CONV_VERSION,
          SCHEMA_NAME,
          TABLE_NAME,
          CONSTRAINT_NAME,
          COLUMN_NAME,
          ORACLE_NAME,
          DELETE_CLAUSE,
          CONSTRAINT_TYPE, 
          DETAIL_ORDER,
          REFCATALOG_NAME,
          REFSCHEMA_NAME,
          REFTABLE_NAME, 
          REFCOLUMN_NAME,
          ISRULE, 
          ISCLUSTEREDINDEX, 
          LANGUAGE,
          COMMENTS,
          CONSTRAINT_NAMESPACE,
          SECURITY_GROUP_ID, 
          CREATED_ON
        FROM 
          MD_CONSTRAINTS 
        WHERE 
          TYPE = 'CAPTURED'
          AND CONSTRAINT_TYPE = 'FOREIGN KEY'
          -- AND SCHEMA_NAME = SCHEMA_NAME
          -- AND TABLE_NAME = TABLE_NAME
          AND PROJECT_ID  = nProjectId
          AND CONNECTION_ID = nConnId
          AND CAPT_VERSION = v_captver
        MINUS
        /*          
        Bug 16817077 - MIGRATION GENERATION : GENERATION OF DUPLICATE FOREIGN KEYS 
        Remove all ForeignKeys that are created on the same table with same referenced Table columns
        4.0 MIGRATION Package : PROCEDURE remove_duplicate_foreignkeys(p_connectionid MD_CONNECTIONS.ID%TYPE)
      */
        SELECT 
          PROJECT_ID, 
          CONNECTION_ID, 
          CATALOG_NAME,
          TYPE,
          CAPT_VERSION,
          CONV_VERSION,
          SCHEMA_NAME,
          TABLE_NAME,
          CONSTRAINT_NAME,
          COLUMN_NAME,
          ORACLE_NAME,
          DELETE_CLAUSE,
          CONSTRAINT_TYPE, 
          DETAIL_ORDER,
          REFCATALOG_NAME,
          REFSCHEMA_NAME,
          REFTABLE_NAME, 
          REFCOLUMN_NAME,
          ISRULE, 
          ISCLUSTEREDINDEX, 
          LANGUAGE,
          COMMENTS,
          CONSTRAINT_NAMESPACE,
          SECURITY_GROUP_ID, 
          CREATED_ON
        FROM (
            SELECT 
              (ROW_NUMBER() OVER (PARTITION BY MC2.COLUMN_NAME, MC2.REFCOLUMN_NAME ORDER BY MC2.CONSTRAINT_NAME)) RN,
              MC2.PROJECT_ID, 
              MC2.CONNECTION_ID, 
              MC2.CATALOG_NAME,
              MC2.TYPE,
              MC2.CAPT_VERSION,
              MC2.CONV_VERSION,
              MC2.SCHEMA_NAME,
              MC2.TABLE_NAME,
              MC2.CONSTRAINT_NAME,
              MC2.COLUMN_NAME,
              MC2.ORACLE_NAME,
              MC2.DELETE_CLAUSE,
              MC2.CONSTRAINT_TYPE, 
              MC2.DETAIL_ORDER,
              MC2.REFCATALOG_NAME,
              MC2.REFSCHEMA_NAME,
              MC2.REFTABLE_NAME, 
              MC2.REFCOLUMN_NAME,
              MC2.ISRULE, 
              MC2.ISCLUSTEREDINDEX, 
              MC2.LANGUAGE,
              MC2.COMMENTS,
              MC2.CONSTRAINT_NAMESPACE,
              MC2.SECURITY_GROUP_ID, 
              MC2.CREATED_ON
            FROM 
              MD_CONSTRAINTS MC1 INNER JOIN
              MD_CONSTRAINTS MC2
            ON
              MC1.CONSTRAINT_TYPE = 'FOREIGN KEY'
              AND MC1.CONSTRAINT_TYPE = MC2.CONSTRAINT_TYPE
              AND MC1.PROJECT_ID = MC2.PROJECT_ID
              AND MC1.CONNECTION_ID = MC2.CONNECTION_ID
              AND MC1.PROJECT_ID  = nProjectId
              AND MC1.CONNECTION_ID = nConnId
              AND MC1.CAPT_VERSION = v_captver
              AND MC1.TYPE = MC2.TYPE
              AND MC1.TYPE = 'CAPTURED'
              AND MC1.SCHEMA_NAME = MC2.SCHEMA_NAME
              AND MC1.TABLE_NAME = MC2.TABLE_NAME
              AND MC1.COLUMN_NAME = MC2.COLUMN_NAME
              AND MC1.REFTABLE_NAME = MC2.REFTABLE_NAME
              AND MC1.REFCOLUMN_NAME = MC2.REFCOLUMN_NAME
              AND MC1.CONSTRAINT_NAME <> MC2.CONSTRAINT_NAME
            )  
        WHERE 
          RN > 1
        )
	);
  COMMIT;
  
  /*
  ADDING MISSING UK'S
  Bug 16817039 - MIGRATION GENERATION : NO MATCHING UNIQUE OR PRIMARY KEY FOR THIS COLUMN-LIST 
  create Uniquekeys where Foreign keys has referened table column does not have UniqueKey
  4.0 MIGRATION Package : PROCEDURE uniquekey_constraint_columns(p_connectionid MD_CONNECTIONS.ID%TYPE)
  
  This is a situation where the Capture has gone bad in SQLServer???
  */
  /*
  INSERT
	INTO MD_STAGE_CONSTRAINTS
	(
	  PROJECT_ID,
	  CONNECTION_ID,
	  CATALOG_NAME,
	  TYPE,
	  SCHEMA_NAME,
	  TABLE_NAME,
	  CONSTRAINT_NAME,
	  COLUMN_NAME,
	  ORACLE_NAME,
	  CONSTRAINT_TYPE,
	  DETAIL_ORDER,
	  ISRULE,
	  ISCLUSTEREDINDEX,
	  LANGUAGE,
	  COMMENTS,
	  CONSTRAINT_NAMESPACE,
	  SECURITY_GROUP_ID, 
	  CREATED_ON
    
--	  CREATED_BY, 
--	  LAST_UPDATED_ON, 
--	  LAST_UPDATED_BY
	)
  (
    SELECT
      PROJECT_ID, 
      CONNECTION_ID, 
      CATALOG_NAME,
      TYPE,
      SCHEMA_NAME,
      TABLE_NAME,
      CONSTRAINT_NAME,
      COLUMN_NAME,
      ORACLE_NAME,
      CONSTRAINT_TYPE,
      DETAIL_ORDER,
      ISRULE, 
      ISCLUSTEREDINDEX, 
      LANGUAGE,
      COMMENTS,
      CONSTRAINT_NAMESPACE,
      SECURITY_GROUP_ID, 
      CREATED_ON
    FROM
       (
       
          SELECT DISTINCT
            FK.PROJECT_ID, 
            FK.CONNECTION_ID, 
            FK.CATALOG_NAME, 
            FK.TYPE, 
            FK.SCHEMA_NAME, 
            FK.REFTABLE_NAME TABLE_NAME, 
            FK.CONSTRAINT_NAME || '_UK' CONSTRAINT_NAME,
            FK.REFCOLUMN_NAME COLUMN_NAME, 
            FK.CONSTRAINT_NAME || '_UK' ORACLE_NAME,
            'UNIQUE' CONSTRAINT_TYPE, 
            (SELECT COLUMN_ORDER FROM MD_COLUMNS WHERE
             PROJECT_ID  = nProjectId
             AND CONNECTION_ID = nConnId
             AND TYPE = 'CAPTURED'
             AND CATALOG_NAME = FK.CATALOG_NAME
             AND SCHEMA_NAME = FK.SCHEMA_NAME
             AND TABLE_NAME = FK.REFTABLE_NAME
             AND COLUMN_NAME = FK.REFCOLUMN_NAME) DETAIL_ORDER,
            FK.ISRULE, 
            FK.ISCLUSTEREDINDEX, 
            FK.LANGUAGE,
            FK.COMMENTS,
            FK.CONSTRAINT_NAMESPACE,
            FK.SECURITY_GROUP_ID, 
            FK.CREATED_ON
          FROM 
            (SELECT *
             FROM 
               MD_CONSTRAINTS 
             WHERE 
               TYPE = 'CAPTURED'
               AND CONSTRAINT_TYPE = 'FOREIGN KEY'
               AND PROJECT_ID  = nProjectId
               AND CONNECTION_ID = nConnId
             ) FK
         WHERE NOT EXISTS (
             SELECT PKUK.CONSTRAINT_NAME 
             FROM 
              (SELECT * 
               FROM MD_CONSTRAINTS 
               WHERE TYPE = 'CAPTURED'
                     AND CONSTRAINT_TYPE = 'PK' OR CONSTRAINT_TYPE = 'UNIQUE'
                     AND PROJECT_ID  = nProjectId
                     AND CONNECTION_ID = nConnId
               ) PKUK 
            WHERE 
              PKUK.TABLE_NAME = FK.REFTABLE_NAME 
              AND PKUK.COLUMN_NAME = FK.REFCOLUMN_NAME
         )
     )
   ); */
   
   INSERT
	INTO MD_STAGE_CONSTRAINTS
	(
	  PROJECT_ID,
	  CONNECTION_ID,
	  CATALOG_NAME,
	  TYPE,
	  CAPT_VERSION,
      CONV_VERSION, 
	  SCHEMA_NAME,
	  TABLE_NAME,
	  CONSTRAINT_NAME,
	  COLUMN_NAME,
	  CONSTRAINT_TYPE,
	  DETAIL_ORDER,
	  ISRULE,
	  ISCLUSTEREDINDEX,
	  LANGUAGE,
	  COMMENTS,
	  CONSTRAINT_NAMESPACE,
	  SECURITY_GROUP_ID, 
	  CREATED_ON
	)
  (
  SELECT 
    FILTER_1.PROJECT_ID,
    FILTER_1.CONNECTION_ID,
    FILTER_1.CATALOG_NAME,
    FILTER_1.TYPE,
    FILTER_1.CAPT_VERSION,
    FILTER_1.CONV_VERSION,
    FILTER_1.SCHEMA_NAME,
    FILTER_1.TABLE_NAME,
    DECODE (
          ROW_NUMBER() OVER (PARTITION BY FILTER_1.RNK ORDER BY FILTER_1.TABLE_NAME),
            1, MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(FILTER_1.TABLE_NAME), '_UK', 128),
            MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(FILTER_1.TABLE_NAME), '_UK', 128)) CONSTRAINT_NAME,
    FILTER_1.COLUMN_NAME, 
    FILTER_1.CONSTRAINT_TYPE,
    FILTER_1.DETAIL_ORDER,
    FILTER_1.ISRULE,
    FILTER_1.ISCLUSTEREDINDEX,
    FILTER_1.LANGUAGE,
    FILTER_1.COMMENTS,
    FILTER_1.CONSTRAINT_NAMESPACE, 
    0, 
    SYSDATE         
  FROM (
  
            SELECT 
                FILTER.PROJECT_ID,
                FILTER.CONNECTION_ID,
                FILTER.CATALOG_NAME,
                FILTER.TYPE,
                FILTER.CAPT_VERSION,
                FILTER.CONV_VERSION,
                FILTER.SCHEMA_NAME,
                FILTER.TABLE_NAME,
                FILTER.CONSTRAINT_NAME,
                FILTER.COLUMN_NAME,
                DENSE_RANK() OVER (PARTITION BY CONSTRAINT_NAMESPACE ORDER BY TABLE_NAME) RNK,
                FILTER.CONSTRAINT_TYPE,
                FILTER.DETAIL_ORDER,
                FILTER.ISRULE,
                FILTER.ISCLUSTEREDINDEX,
                FILTER.LANGUAGE,
                FILTER.COMMENTS,
                FILTER.CONSTRAINT_NAMESPACE  
            FROM 
            (
                SELECT FK.PROJECT_ID,
                    FK.CONNECTION_ID,
                    FK.CATALOG_NAME,
                    FK.TYPE,
                    FK.CAPT_VERSION,
                    FK.CONV_VERSION,
                    FK.SCHEMA_NAME,
                    FK.reftable_name TABLE_NAME,
                    MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(FK.reftable_name||'_'||FK.refcolumn_name), '_UK', 128) CONSTRAINT_NAME,
                    FK.refcolumn_name COLUMN_NAME,
                    'UNIQUE' CONSTRAINT_TYPE,
                    (SELECT COLUMN_ORDER 
                     FROM MD_COLUMNS 
                     WHERE PROJECT_ID  = nProjectId
                     AND CONNECTION_ID = nConnId
                     AND CAPT_VERSION = v_captver 
                     AND TYPE          = 'CAPTURED'
                     AND CATALOG_NAME  = FK.CATALOG_NAME
                     AND SCHEMA_NAME   = FK.SCHEMA_NAME
                     AND TABLE_NAME    = FK.REFTABLE_NAME
                     AND COLUMN_NAME   = FK.REFCOLUMN_NAME
                    ) DETAIL_ORDER,
                    'N' ISRULE,
                    'N' ISCLUSTEREDINDEX,
                    (SELECT DISTINCT LANGUAGE FROM MD_CONSTRAINTS ) LANGUAGE,
                    NULL COMMENTS,
                    'NS_CONSTRAINTS-'|| FK.SCHEMA_NAME|| '-'|| FK.CONNECTION_ID|| '-'|| FK.PROJECT_ID CONSTRAINT_NAMESPACE
                  FROM
                    (SELECT *
                    FROM
                        MD_CONSTRAINTS
                      -- MD_STAGE_CONSTRAINTS
                    WHERE TYPE          = 'CAPTURED'
                    AND CONSTRAINT_TYPE = 'FOREIGN KEY'
                    AND PROJECT_ID      = nProjectId
                    AND CONNECTION_ID   = nConnId
                    AND CAPT_VERSION = v_captver
                    
                    ) FK
                  WHERE NOT EXISTS
                    (SELECT PKUK.CONSTRAINT_NAME
                    FROM
                      (SELECT *
                         FROM MD_CONSTRAINTS
                      -- FROM MD_STAGE_CONSTRAINTS
                      WHERE TYPE          = 'CAPTURED'
                      AND CONSTRAINT_TYPE = 'PK'
                      OR CONSTRAINT_TYPE  = 'UNIQUE'
                      AND PROJECT_ID      = nProjectId
                      AND CONNECTION_ID   = nConnId
                      AND CAPT_VERSION = v_captver
                      ) PKUK
                    WHERE PKUK.TABLE_NAME = FK.REFTABLE_NAME
                    AND PKUK.COLUMN_NAME  = FK.REFCOLUMN_NAME
                    )
                  GROUP BY FK.PROJECT_ID,
                    FK.CONNECTION_ID,
                    FK.CATALOG_NAME,
                    FK.TYPE,
                    FK.CAPT_VERSION,
                    FK.CONV_VERSION,
                    FK.SCHEMA_NAME,
                    FK.REFTABLE_NAME,
                    FK.REFCOLUMN_NAME,
                    FK.DETAIL_ORDER 
                  ORDER BY  
                    FK.PROJECT_ID,
                    FK.CONNECTION_ID,
                    FK.CATALOG_NAME,
                    FK.SCHEMA_NAME,
                    FK.REFTABLE_NAME,
                    FK.DETAIL_ORDER 
            ) FILTER
  ) FILTER_1
  );
   
  COMMIT;
  
  END ConstraintCleanup;

  PROCEDURE ConvertConstraints(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  v_andor CLOB;
  cur NUMBER;
  ex NUMBER;
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);	  
  ConstraintCleanup(nProjectId, nConnId, v_srcdatabase);
  v_sql := q'{
  INSERT
  INTO MD_CONSTRAINTS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
	  CONV_VERSION,
      SCHEMA_NAME,
      TABLE_NAME,
      CONSTRAINT_NAME,
      COLUMN_NAME,
      ORACLE_NAME,
      DELETE_CLAUSE,
      CONSTRAINT_TYPE,
      DETAIL_ORDER,
      REFCATALOG_NAME,
      REFSCHEMA_NAME,
      REFTABLE_NAME,
      REFCOLUMN_NAME,
      CONSTRAINT_TEXT,
      ISRULE,
      ISCLUSTEREDINDEX,
      LANGUAGE,
      COMMENTS,
      CONSTRAINT_NAMESPACE
    )
    (
        /*
           Doing a final check on Duplicate ORACLE_NAME.
           Modify ORACLE_NAME's whose RANK > 1. 
         */
--        SELECT 
--		    RNKTAB2.PROJECT_ID,
--		    RNKTAB2.CONNECTION_ID,
--		    RNKTAB2.CATALOG_NAME,
--		    RNKTAB2.TYPE,
--		    RNKTAB2.SCHEMA_NAME,
--		    RNKTAB2.TABLE_NAME, 
--		    RNKTAB2.CONSTRAINT_NAME,
--		    RNKTAB2.COLUMN_NAME,
--		    DECODE(RNKTAB2.RNK, 
--						1, MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(RNKTAB2.ORACLE_NAME), -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and Rank for that row now. 
--						MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(RNKTAB2.ORACLE_NAME)), '_' || TO_CHAR(RNKTAB2.RNK), 128)
--				  ) ORACLE_NAME,
--		    
--		    RNKTAB2.DELETE_CLAUSE,
--		    RNKTAB2.CONSTRAINT_TYPE,
--		    RNKTAB2.DETAIL_ORDER,
--		    RNKTAB2.REFCATALOG_NAME,
--		    RNKTAB2.REFSCHEMA_NAME,
--		    RNKTAB2.REFTABLE_NAME,
--		    RNKTAB2.REFCOLUMN_NAME,
--		    RNKTAB2.CONSTRAINT_TEXT,
--		    RNKTAB2.ISRULE,
--		    RNKTAB2.ISCLUSTEREDINDEX,
--		    RNKTAB2.LANGUAGE,
--		    RNKTAB2.COMMENTS,
--		    RNKTAB2.CONSTRAINT_NAMESPACE
--		    
--		    FROM 
--		    (
--		       /*
--		          RANK the ORACLE_NAME to see if we end up with Duplicates with RANK > 1.
--		        */
--		       SELECT 
--		          RNKTAB1.PROJECT_ID,
--		          RNKTAB1.CONNECTION_ID,
--		          RNKTAB1.CATALOG_NAME,
--		          RNKTAB1.TYPE,
--		          RNKTAB1.SCHEMA_NAME,
--		          RNKTAB1.TABLE_NAME, 
--		          RNKTAB1.CONSTRAINT_NAME,
--		          RNKTAB1.COLUMN_NAME,
--		          DENSE_RANK() OVER (PARTITION BY RNKTAB1.CONSTRAINT_NAMESPACE, RNKTAB1.ORACLE_NAME ORDER BY RNKTAB1.TABLE_NAME) RNK,
--        		  RNKTAB1.ORACLE_NAME,
--		          RNKTAB1.DELETE_CLAUSE,
--		          RNKTAB1.CONSTRAINT_TYPE,
--		          RNKTAB1.DETAIL_ORDER,
--		          RNKTAB1.REFCATALOG_NAME,
--		          RNKTAB1.REFSCHEMA_NAME,
--		          RNKTAB1.REFTABLE_NAME,
--		          RNKTAB1.REFCOLUMN_NAME,
--		          RNKTAB1.CONSTRAINT_TEXT,
--		          RNKTAB1.ISRULE,
--		          RNKTAB1.ISCLUSTEREDINDEX,
--		          RNKTAB1.LANGUAGE,
--		          RNKTAB1.COMMENTS,
--		          RNKTAB1.CONSTRAINT_NAMESPACE
--		          
--		       FROM 
--	            (
--	             /*
--	                Modify CONSTRAINT_NAME whose Ranks are > 1. 
--	              */
--	             SELECT 
--	                RNKTAB.PROJECT_ID,
--	                RNKTAB.CONNECTION_ID,
--	                RNKTAB.CATALOG_NAME,
--	                'CONVERTED' TYPE,
--	                RNKTAB.SCHEMA_NAME,
--	                RNKTAB.TABLE_NAME, 
--	                RNKTAB.CONSTRAINT_NAME,
--	                RNKTAB.COLUMN_NAME,
--	                DECODE(RNKTAB.RNK, 
--								1, MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(CONSTRAINT_NAME), -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and Rank for that row now. 
--								MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(CONSTRAINT_NAME)), '_' || TO_CHAR(RNKTAB.RNK), 128)
--                          ) ORACLE_NAME,        
--	                RNKTAB.DELETE_CLAUSE,
--	                RNKTAB.CONSTRAINT_TYPE,
--	                RNKTAB.DETAIL_ORDER,
--	                RNKTAB.REFCATALOG_NAME,
--	                RNKTAB.REFSCHEMA_NAME,
--	                RNKTAB.REFTABLE_NAME,
--	                RNKTAB.REFCOLUMN_NAME,
--	                RNKTAB.CONSTRAINT_TEXT,
--	                RNKTAB.ISRULE,
--	                RNKTAB.ISCLUSTEREDINDEX,
--	                RNKTAB.LANGUAGE,
--	                RNKTAB.COMMENTS,
--	                RNKTAB.CONSTRAINT_NAMESPACE
--	                FROM 
--	                (
--	                  /*
--	                    Constraints can have Composite keys so let's Rank the CONSTRAINT_NAME's within CONSTRAINT_NAMESPACE. 
--	                    Duplicates will have RANK > 1 (we use DENSE_RANK here).
--	                  */
--	                  SELECT 
--	                      PROJECT_ID,
--	                      CONNECTION_ID,
--	                      CATALOG_NAME,
--	                      TYPE,
--	                      SCHEMA_NAME,
--	                      TABLE_NAME, 
--	                      CONSTRAINT_NAME,
--	                      COLUMN_NAME,
--	                      DENSE_RANK() OVER (PARTITION BY CONSTRAINT_NAMESPACE, CONSTRAINT_NAME ORDER BY TABLE_NAME) RNK,
--	                      DELETE_CLAUSE,
--	                      CONSTRAINT_TYPE,
--	                      DETAIL_ORDER,
--	                      REFCATALOG_NAME,
--	                      REFSCHEMA_NAME,
--	                      (SELECT DISTINCT 
--	                          t.ORACLE_NAME
--	                       FROM 
--	                          MD_TABLES t
--	                       WHERE 
--	                          t.PROJECT_ID = nProjectId 
--          	                  AND t.CONNECTION_ID = nConnId 
--	                          AND t.CATALOG_NAME = c.REFCATALOG_NAME
--	                          AND t.SCHEMA_NAME = c.REFSCHEMA_NAME
--	                          AND t.TABLE_NAME = c.REFTABLE_NAME
--	                          AND c.CONSTRAINT_TYPE = 'FOREIGN KEY'
--	                          AND TYPE = 'CONVERTED'
--	                      ) REFTABLE_NAME,
--	                      (SELECT DISTINCT 
--	                          t.ORACLE_NAME 
--	                       FROM 
--	                          MD_COLUMNS t 
--	                       WHERE 
--	                          t.PROJECT_ID = nProjectId 
--          	                  AND t.CONNECTION_ID = nConnId 
--	                          AND t.CATALOG_NAME = c.REFCATALOG_NAME 
--	                          AND t.SCHEMA_NAME = c.REFSCHEMA_NAME 
--	                          AND t.TABLE_NAME = c.REFTABLE_NAME
--	                          AND t.COLUMN_NAME = c.REFCOLUMN_NAME
--	                          AND c.CONSTRAINT_TYPE = 'FOREIGN KEY'
--	                          AND t.TYPE = 'CONVERTED'
--	                       ) REFCOLUMN_NAME,
--	                      CONSTRAINT_TEXT,
--	                      ISRULE,
--	                      ISCLUSTEREDINDEX,
--	                      LANGUAGE,
--	                      COMMENTS,
--	                      CONSTRAINT_NAMESPACE
--	                 FROM 
--	                    MD_STAGE_CONSTRAINTS c  
--	                 WHERE 
--	                    PROJECT_ID  = nProjectId
--	                    AND CONNECTION_ID = nConnId
--	                    AND TYPE = 'CAPTURED'
--	                ) RNKTAB
--		        ) RNKTAB1  
--		   ) RNKTAB2 

		SELECT 
		  RNKTAB.PROJECT_ID,
		  RNKTAB.CONNECTION_ID,
		  RNKTAB.CATALOG_NAME,
		  'CONVERTED' TYPE,
		  :captvId CAPT_VERSION,
		  :verId CONV_VERSION, 
		  RNKTAB.SCHEMA_NAME,
		  RNKTAB.TABLE_NAME, 
		  RNKTAB.CONSTRAINT_NAME,
		  RNKTAB.COLUMN_NAME,
		  DECODE(RNKTAB.RNK, 
		  1, MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(CONSTRAINT_NAME), -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and Rank for that row now. 
		  MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(CONSTRAINT_NAME)), '_C' || TO_CHAR(RNKTAB.RNK-1), 128)
		        ) ORACLE_NAME,        
		  RNKTAB.DELETE_CLAUSE,
		  RNKTAB.CONSTRAINT_TYPE,
		  RNKTAB.DETAIL_ORDER,
		  RNKTAB.REFCATALOG_NAME,
		  RNKTAB.REFSCHEMA_NAME,
		  RNKTAB.REFTABLE_NAME,
		  RNKTAB.REFCOLUMN_NAME,
		  RNKTAB.CONSTRAINT_TEXT,
		  RNKTAB.ISRULE,
		  RNKTAB.ISCLUSTEREDINDEX,
		  RNKTAB.LANGUAGE,
		  RNKTAB.COMMENTS,
		  RNKTAB.CONSTRAINT_NAMESPACE
		FROM 
		    (
		      /*
		    Constraints can have Composite keys so let's Rank the CONSTRAINT_NAME's within CONSTRAINT_NAMESPACE. 
		    Duplicates will have RANK > 1 (we use DENSE_RANK here).
		  */
		  SELECT 
		      PROJECT_ID,
		      CONNECTION_ID,
		      CATALOG_NAME,
		      TYPE,
		      SCHEMA_NAME,
		      TABLE_NAME, 
		      CONSTRAINT_NAME,
		      COLUMN_NAME,
		      --DENSE_RANK() OVER (PARTITION BY CONSTRAINT_NAMESPACE, UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(CONSTRAINT_NAME)) ORDER BY UPPER(CONSTRAINT_NAME), TABLE_NAME, CONSTRAINT_NAMESPACE) RNK,
		      DENSE_RANK() OVER (PARTITION BY CONSTRAINT_NAMESPACE, CONSTRAINT_NAME ORDER BY UPPER(CONSTRAINT_NAME), TABLE_NAME, CONSTRAINT_NAMESPACE) RNK,
		      DELETE_CLAUSE,
		      CONSTRAINT_TYPE,
		      DETAIL_ORDER,
		      REFCATALOG_NAME,
		      REFSCHEMA_NAME,
		      (SELECT DISTINCT 
		          t.ORACLE_NAME
		       FROM 
		          MD_TABLES t
		       WHERE 
		          t.PROJECT_ID = :projectId 
		          AND t.CONNECTION_ID = :svrId 
		          AND t.CATALOG_NAME = c.REFCATALOG_NAME
		          AND t.SCHEMA_NAME = c.REFSCHEMA_NAME
		          AND t.TABLE_NAME = c.REFTABLE_NAME
		          AND c.CONSTRAINT_TYPE = 'FOREIGN KEY'
		          AND TYPE = 'CONVERTED'
		      ) REFTABLE_NAME,
		      (SELECT DISTINCT 
		          t.ORACLE_NAME 
		       FROM 
		          MD_COLUMNS t 
		       WHERE 
		          t.PROJECT_ID = :projectId 
		          AND t.CONNECTION_ID = :svrId 
		          AND t.CATALOG_NAME = c.REFCATALOG_NAME 
		          AND t.SCHEMA_NAME = c.REFSCHEMA_NAME 
		          AND t.TABLE_NAME = c.REFTABLE_NAME
		          AND t.COLUMN_NAME = c.REFCOLUMN_NAME
		          AND c.CONSTRAINT_TYPE = 'FOREIGN KEY'
		          AND t.TYPE = 'CONVERTED'
		       ) REFCOLUMN_NAME,
		      CONSTRAINT_TEXT,
		      ISRULE,
		      ISCLUSTEREDINDEX,
		      LANGUAGE,
		      COMMENTS,
		      CONSTRAINT_NAMESPACE
		 FROM 
		    MD_STAGE_CONSTRAINTS c  
		 WHERE 
		    PROJECT_ID  = :projectId
		    AND CONNECTION_ID = :svrId
		    AND CAPT_VERSION = :captvId
		    AND TYPE = 'CAPTURED'
	}';    
		    
	cur := DBMS_SQL.OPEN_CURSOR;
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;

    v_schemas := convertArrToStr(schemaArr);
	v_tables := convertArrToStr(tableArr);
	
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	   v_andor := q'{ AND ((CONSTRAINT_TYPE = 'FOREIGN KEY' AND UPPER(REFSCHEMA_NAME) IN(}' || v_schemas || ')' || ' AND UPPER(REFTABLE_NAME) IN(' || v_tables || ')) OR ' ||  
	           q'{CONSTRAINT_TYPE = 'PK' OR CONSTRAINT_TYPE = 'CHECK')}';
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0 THEN
	   v_andor := q'{ AND ((CONSTRAINT_TYPE = 'FOREIGN KEY' AND UPPER(REFSCHEMA_NAME) IN(}' || v_schemas || ')) OR ' ||  
	           q'{CONSTRAINT_TYPE = 'PRIMARY KEY' OR CONSTRAINT_TYPE = 'CHECK')}';
	ELSE
	   v_andor := EMPTY_CLOB();
	END IF; 


    v_sql := v_sql || v_andor || ') RNKTAB)';
	
    -- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
    DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_CONSTRAINTS.');
    DBMS_SQL.CLOSE_CURSOR(cur);
	COMMIT;
  END ConvertConstraints;

  PROCEDURE IndexCleanup(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2) AS

BEGIN
 
  INSERT
  INTO MD_STAGE_INDEXES
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      SCHEMA_NAME,
      TABLE_NAME,
      COLUMN_NAME,
      INDEX_NAME,
      INDEX_TYPE,
      INDEX_PORTION,
      IS_DESCENDING,
      DETAIL_ORDER,
      ISCLUSTEREDINDEX,
      COMMENTS,
      ORACLE_NAME,
      INDEX_NAMESPACE
    )
    (
        /*
        Bug 17345898 - MYSQL:NAMING CONFLICT IN GENERATED DDL 
        MIGRATION Package : PROCEDURE remove_duplicate_id_objecttype
        Check the Unique Key and PK on a table and table column, check if there is an index, if there is one do not copy that Index
        */
  /*
        SELECT DISTINCT
          INDX.PROJECT_ID,
          INDX.CONNECTION_ID, 
          INDX.CATALOG_NAME, 
          INDX.TYPE, 
          INDX.SCHEMA_NAME, 
          INDX.TABLE_NAME, 
          INDX.COLUMN_NAME, 
          INDX.INDEX_NAME, 
          INDX.INDEX_TYPE, 
          INDX.INDEX_PORTION,
          INDX.IS_DESCENDING,
          INDX.DETAIL_ORDER,
          INDX.ISCLUSTEREDINDEX,
          INDX.COMMENTS,
          INDX.ORACLE_NAME,
          INDX.INDEX_NAMESPACE
        FROM 
          MD_INDEXES INDX, MD_CONSTRAINTS CONSTR
        WHERE
          CONSTR.PROJECT_ID  = nProjectId
          AND CONSTR.CONNECTION_ID = nConnId
          AND CONSTR.CAPT_VERSION = v_captver
          AND (CONSTR.CONSTRAINT_TYPE = 'PK' OR CONSTR.CONSTRAINT_TYPE = 'UNIQUE') 
          AND CONSTR.PROJECT_ID = INDX.PROJECT_ID
          AND CONSTR.CONNECTION_ID = INDX.CONNECTION_ID
          AND CONSTR.TYPE = INDX.TYPE
          AND CONSTR.TYPE = 'CAPTURED'
          AND CONSTR.SCHEMA_NAME = INDX.SCHEMA_NAME
 --         AND CONSTR.TABLE_NAME = INDX.TABLE_NAME
          AND CONSTR.CONSTRAINT_NAME <> INDX.INDEX_NAME
 --         AND INDX.TYPE = 'CAPTURED'
 --         AND CONSTR.TYPE = 'CONVERTED'
 --         AND CONSTR.ORACLE_NAME <> INDX.INDEX_NAME
 
   */
  SELECT 
  		RNKTAB.PROJECT_ID,
        RNKTAB.CONNECTION_ID, 
        RNKTAB.CATALOG_NAME, 
        RNKTAB.TYPE,
        RNKTAB.CAPT_VERSION,
        RNKTAB.CONV_VERSION, 
        RNKTAB.SCHEMA_NAME, 
        RNKTAB.TABLE_NAME, 
        RNKTAB.COLUMN_NAME, 
        RNKTAB.INDEX_NAME,
        RNKTAB.INDEX_TYPE, 
        RNKTAB.INDEX_PORTION,
        RNKTAB.IS_DESCENDING,
        RNKTAB.DETAIL_ORDER,
        RNKTAB.ISCLUSTEREDINDEX,
        RNKTAB.COMMENTS,
        RNKTAB.ORACLE_NAME,
        RNKTAB.INDEX_NAMESPACE
  FROM
  ( 
    SELECT DISTINCT
        FILTER.PROJECT_ID,
        FILTER.CONNECTION_ID, 
        FILTER.CATALOG_NAME, 
        FILTER.TYPE,
        FILTER.CAPT_VERSION,
        FILTER.CONV_VERSION, 
        FILTER.SCHEMA_NAME, 
        FILTER.TABLE_NAME, 
        FILTER.COLUMN_NAME, 
        FILTER.INDEX_NAME,
        DENSE_RANK() OVER (PARTITION BY FILTER.TABLE_NAME, FILTER.COLUMN_NAME  ORDER BY UPPER(INDEX_NAME), FILTER.TABLE_NAME, FILTER.COLUMN_NAME) RNK, 
        FILTER.INDEX_TYPE, 
        FILTER.INDEX_PORTION,
        FILTER.IS_DESCENDING,
        FILTER.DETAIL_ORDER,
        FILTER.ISCLUSTEREDINDEX,
        FILTER.COMMENTS,
        FILTER.ORACLE_NAME,
        FILTER.INDEX_NAMESPACE
    FROM  ( 
              SELECT DISTINCT
                            INDX.PROJECT_ID,
                            INDX.CONNECTION_ID, 
                            INDX.CATALOG_NAME, 
                            INDX.TYPE,
        					INDX.CAPT_VERSION,
        					INDX.CONV_VERSION,  
                            INDX.SCHEMA_NAME, 
                            INDX.TABLE_NAME, 
                            INDX.COLUMN_NAME, 
                            INDX.INDEX_NAME, 
                            INDX.INDEX_TYPE, 
                            INDX.INDEX_PORTION,
                            INDX.IS_DESCENDING,
                            INDX.DETAIL_ORDER,
                            INDX.ISCLUSTEREDINDEX,
                            INDX.COMMENTS,
                            INDX.ORACLE_NAME,
                            INDX.INDEX_NAMESPACE
                          FROM 
                            MD_INDEXES INDX, MD_CONSTRAINTS CONSTR
                          WHERE
                            CONSTR.PROJECT_ID  = nProjectId
                            AND CONSTR.CONNECTION_ID = nConnId
                			AND CONSTR.CAPT_VERSION = v_captver
                            AND (CONSTR.CONSTRAINT_TYPE = 'PK' OR CONSTR.CONSTRAINT_TYPE = 'UNIQUE') 
                            AND CONSTR.PROJECT_ID = INDX.PROJECT_ID
                            AND CONSTR.CONNECTION_ID = INDX.CONNECTION_ID
                            AND CONSTR.TYPE = INDX.TYPE
                            AND CONSTR.TYPE = 'CAPTURED'
                            AND CONSTR.SCHEMA_NAME = INDX.SCHEMA_NAME
                            AND CONSTR.TABLE_NAME = INDX.TABLE_NAME
                            AND CONSTR.CONSTRAINT_NAME <> INDX.INDEX_NAME
          ) FILTER
    WHERE NOT EXISTS (
            SELECT 
                FILTER.PROJECT_ID,
                FILTER.CONNECTION_ID, 
                FILTER.CATALOG_NAME, 
                FILTER.TYPE,
                FILTER.CAPT_VERSION,
        		FILTER.CONV_VERSION, 
                FILTER.SCHEMA_NAME, 
                FILTER.TABLE_NAME, 
                FILTER.COLUMN_NAME, 
                FILTER.INDEX_NAME, 
                FILTER.INDEX_TYPE, 
                FILTER.INDEX_PORTION,
                FILTER.IS_DESCENDING,
                FILTER.DETAIL_ORDER,
                FILTER.ISCLUSTEREDINDEX,
                FILTER.COMMENTS,
                FILTER.ORACLE_NAME,
                FILTER.INDEX_NAMESPACE
             FROM 
               MD_CONSTRAINTS CONS
             WHERE
                CONS.PROJECT_ID  = nProjectId
                AND CONS.CONNECTION_ID = nConnId
                AND CONS.CAPT_VERSION = v_captver
                AND (CONS.CONSTRAINT_TYPE = 'PK' OR CONS.CONSTRAINT_TYPE = 'UNIQUE') 
                AND CONS.PROJECT_ID = FILTER.PROJECT_ID
                AND CONS.CONNECTION_ID = FILTER.CONNECTION_ID
                AND CONS.TYPE = FILTER.TYPE
                AND CONS.TYPE = 'CAPTURED'
                AND CONS.SCHEMA_NAME = FILTER.SCHEMA_NAME
                AND CONS.CONSTRAINT_NAME = FILTER.INDEX_NAME
        ) 
       ) RNKTAB
       WHERE RNK = 1 
      );
    COMMIT;

END IndexCleanup;
  PROCEDURE ConvertIndexes(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;
  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);
    -- This needs to be tested on a MySQL testcase before enabling this function
  IndexCleanup(nProjectId, nConnId, v_srcdatabase);
    
  v_sql := q'{
  INSERT
  INTO MD_INDEXES
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      SCHEMA_NAME,
      TABLE_NAME,
      COLUMN_NAME,
      INDEX_NAME,
      INDEX_TYPE,
      INDEX_PORTION,
      IS_DESCENDING,
      DETAIL_ORDER,
      NATIVE_SQL,
      NATIVE_KEY,
      ISCLUSTEREDINDEX,
      COMMENTS,
      ORACLE_NAME,
      INDEX_NAMESPACE
    )
    /*(
       SELECT
          RNKTAB2.PROJECT_ID,
          RNKTAB2.CONNECTION_ID,
          RNKTAB2.CATALOG_NAME,
          RNKTAB2.TYPE,
          RNKTAB2.SCHEMA_NAME,
          RNKTAB2.TABLE_NAME,
          RNKTAB2.COLUMN_NAME,
          RNKTAB2.INDEX_NAME,
          RNKTAB2.INDEX_TYPE,
          RNKTAB2.INDEX_PORTION,
          RNKTAB2.IS_DESCENDING,
          RNKTAB2.DETAIL_ORDER,
          RNKTAB2.NATIVE_SQL,
          RNKTAB2.NATIVE_KEY,
          RNKTAB2.ISCLUSTEREDINDEX,
          RNKTAB2.COMMENTS,
          DECODE(RNKTAB2.RNK, 
				1, MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(RNKTAB2.ORACLE_NAME), -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and Rank for that row now. 
				MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(RNKTAB2.ORACLE_NAME)), '_' || TO_CHAR(RNKTAB2.RNK), 128)
		  ) ORACLE_NAME,
		  RNKTAB2.INDEX_NAMESPACE          
       FROM 
       (
              SELECT
	              RNKTAB1.PROJECT_ID,
		          RNKTAB1.CONNECTION_ID,
		          RNKTAB1.CATALOG_NAME,
		          RNKTAB1.TYPE,
		          RNKTAB1.SCHEMA_NAME,
		          RNKTAB1.TABLE_NAME,
		          RNKTAB1.COLUMN_NAME,
		          RNKTAB1.INDEX_NAME,
		          RNKTAB1.INDEX_TYPE,
		          RNKTAB1.INDEX_PORTION,
		          RNKTAB1.IS_DESCENDING,
		          RNKTAB1.DETAIL_ORDER,
		          RNKTAB1.NATIVE_SQL,
		          RNKTAB1.NATIVE_KEY,
		          RNKTAB1.ISCLUSTEREDINDEX,
		          RNKTAB1.COMMENTS,
		          RNKTAB1.ORACLE_NAME,
		          DENSE_RANK() OVER (PARTITION BY RNKTAB1.INDEX_NAMESPACE, RNKTAB1.ORACLE_NAME ORDER BY RNKTAB1.TABLE_NAME) RNK, 
		          RNKTAB1.INDEX_NAMESPACE
		       FROM 
	           (
				       SELECT  
				          RNKTAB.PROJECT_ID,
				          RNKTAB.CONNECTION_ID,
				          RNKTAB.CATALOG_NAME,
				          RNKTAB.TYPE,
				          RNKTAB.SCHEMA_NAME,
				          RNKTAB.TABLE_NAME,
				          RNKTAB.COLUMN_NAME,
				          RNKTAB.INDEX_NAME,
				          RNKTAB.INDEX_TYPE,
				          RNKTAB.INDEX_PORTION,
				          RNKTAB.IS_DESCENDING,
				          RNKTAB.DETAIL_ORDER,
				          RNKTAB.NATIVE_SQL,
				          RNKTAB.NATIVE_KEY,
				          RNKTAB.ISCLUSTEREDINDEX,
				          RNKTAB.COMMENTS,
				          DECODE(RNKTAB.RNK, 
												1, MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(INDEX_NAME), -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and Rank for that row now. 
												MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(INDEX_NAME)), '_' || TO_CHAR(RNKTAB.RNK), 128)
				                          ) ORACLE_NAME,     
				          RNKTAB.INDEX_NAMESPACE
				        FROM
				          (SELECT 
				            PROJECT_ID,
				            CONNECTION_ID,
				            CATALOG_NAME,
				            'CONVERTED' TYPE,
				            SCHEMA_NAME,
				            TABLE_NAME,
				            COLUMN_NAME,
				            INDEX_NAME,
				            INDEX_TYPE,
				            INDEX_PORTION,
				            IS_DESCENDING,
				            DETAIL_ORDER,
				            NATIVE_SQL,
				            NATIVE_KEY,
				            ISCLUSTEREDINDEX,
				            COMMENTS,
				            DENSE_RANK() OVER (PARTITION BY INDEX_NAMESPACE, INDEX_NAME ORDER BY TABLE_NAME) RNK,				            
				            INDEX_NAMESPACE
				          FROM 
				             MD_INDEXES 
				             -- MD_STAGE_INDEXES
				          WHERE 
				             PROJECT_ID  = nProjectId
				             AND CONNECTION_ID = nConnId
				             AND TYPE = 'CAPTURED'
				          ) RNKTAB
				     )RNKTAB1
	         ) RNKTAB2			     
    )*/
    
    (
        SELECT
          RNKTAB.PROJECT_ID,
          RNKTAB.CONNECTION_ID,
          RNKTAB.CATALOG_NAME,
          RNKTAB.TYPE,
          :captvId CAPT_VERSION,
          :verId CONV_VERSION,
          RNKTAB.SCHEMA_NAME,
          RNKTAB.TABLE_NAME,
          RNKTAB.COLUMN_NAME,
          RNKTAB.INDEX_NAME,
          RNKTAB.INDEX_TYPE,
          RNKTAB.INDEX_PORTION,
          RNKTAB.IS_DESCENDING,
          RNKTAB.DETAIL_ORDER,
          RNKTAB.NATIVE_SQL,
          RNKTAB.NATIVE_KEY,
          RNKTAB.ISCLUSTEREDINDEX,
          RNKTAB.COMMENTS,
          DECODE(RNKTAB.RNK, 
                1, MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(INDEX_NAME), -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and Rank for that row now. 
                MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(INDEX_NAME)), '_I' || TO_CHAR(RNKTAB.RNK-1), 128)
                ) ORACLE_NAME,     
          RNKTAB.INDEX_NAMESPACE
        FROM
          (SELECT 
            PROJECT_ID,
            CONNECTION_ID,
            CATALOG_NAME,
            'CONVERTED' TYPE,
            SCHEMA_NAME,
            TABLE_NAME,
            COLUMN_NAME,
            INDEX_NAME,
            INDEX_TYPE,
            INDEX_PORTION,
            IS_DESCENDING,
            DETAIL_ORDER,
            NATIVE_SQL,
            NATIVE_KEY,
            ISCLUSTEREDINDEX,
            COMMENTS,
            DENSE_RANK() OVER (PARTITION BY INDEX_NAMESPACE, INDEX_NAME ORDER BY INDEX_NAME, TABLE_NAME, INDEX_NAMESPACE) RNK,				            
            INDEX_NAMESPACE
          FROM 
             -- MD_INDEXES 
             MD_STAGE_INDEXES
          WHERE 
             PROJECT_ID  = :projectId
             AND CONNECTION_ID = :svrId
             AND CAPT_VERSION = :captvId 
             AND TYPE = 'CAPTURED'
     }';       
     
    cur := DBMS_SQL.OPEN_CURSOR;
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;
    v_sql := v_sql || ') RNKTAB)';
	
    -- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
    DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_INDEXES.');
    DBMS_SQL.CLOSE_CURSOR(cur);        
   COMMIT;
  END ConvertIndexes;

  PROCEDURE ConvertViews(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB;
  cur NUMBER;
  ex NUMBER;
  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  
  v_sql := q'{
	  INSERT
	  INTO MD_VIEWS
	    (
	      PROJECT_ID,
	      CONNECTION_ID,
	      CATALOG_NAME,
	      TYPE,
	      CAPT_VERSION,
	      CONV_VERSION,
	      SCHEMA_NAME,
	      VIEW_NAME,
	      ORACLE_NAME,
	      NATIVE_SQL,
	      NATIVE_KEY,
	      LANGUAGE,
	      COMMENTS,
	      LINECOUNT,
	      SCHEMA_NAMESPACE
	    )
	    (SELECT
	        SRC.PROJECT_ID,
	        SRC.CONNECTION_ID,
	        SRC.CATALOG_NAME,
	        SRC.TYPE,
	        :captvId CAPT_VERSION,
	        :verId CONV_VERSION,
	        SRC.SCHEMA_NAME,
	        SRC.VIEW_NAME,
	        TAB.ORACLE_NAME,
	        SRC.NATIVE_SQL,
	        SRC.NATIVE_KEY,
	        SRC.LANGUAGE,
	        SRC.COMMENTS,
	        SRC.LINECOUNT,
	        SRC.SCHEMA_NAMESPACE
	      FROM
	        (SELECT ROWID RID,
	          PROJECT_ID,
	          CONNECTION_ID,
	          CATALOG_NAME,
	          'CONVERTED' TYPE,
	          SCHEMA_NAME,
	          VIEW_NAME,
	          SCHEMA_NAMESPACE,
	          ORACLE_NAME,
	          NATIVE_SQL,
	          NATIVE_KEY,
	          LANGUAGE,
	          COMMENTS,
	          LINECOUNT
	        FROM MD_VIEWS
	        WHERE PROJECT_ID  = :projectId
	        AND CONNECTION_ID = :svrId
	        AND CAPT_VERSION = :captvId
	        AND TYPE          = 'CAPTURED'
	    }';

   v_sql1 := q'{
	    ) SRC,
        (SELECT DECODE( (ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 
           1, UTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
           MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(UTAB.ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 128) ) ORACLE_NAME,
          UTAB.RID RID
        FROM
          (SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_TABLES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(VIEW_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_VIEWS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SEQUENCE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SEQUENCES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PROGRAM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_STORED_PROGRAMS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PACKAGE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_PACKAGES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(DATA_TYPE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_USER_DEFINED_DATA_TYPES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SYNONYM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SYNONYMS
          WHERE 
          PRIVATE_VISIBILITY = 'Y'
          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          ) UTAB
        ) TAB
      WHERE SRC.RID = TAB.RID
    )
   }'; 

   cur := DBMS_SQL.OPEN_CURSOR;
   IF DBMS_LOB.GETLENGTH(v_schemas) > 0  THEN
	 v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	 v_sql := v_sql || v_schemas || v_sql1;
  ELSE
	 v_sql := v_sql || v_sql1;
  END IF;
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);
  DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
  DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
  DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
  DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
  DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
  ex := DBMS_SQL.EXECUTE(cur);
  DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_VIEWS.');
  DBMS_SQL.CLOSE_CURSOR(cur);    
  COMMIT;
  END ConvertViews;

  PROCEDURE ConvertStoredPrograms(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB;
  cur NUMBER;
  ex NUMBER;
  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_sql := q'{
    INSERT
	  INTO MD_STORED_PROGRAMS
	    (
	      PROJECT_ID,
	      CONNECTION_ID,
	      CATALOG_NAME,
	      TYPE,
	      CAPT_VERSION,
	      CONV_VERSION,
	      SCHEMA_NAME,
	      PROGRAM_NAME,
	      ORACLE_NAME,
	      PROGRAMTYPE,
	      FUNCTIONTYPE,
	      NATIVE_SQL,
	      NATIVE_KEY,
	      LANGUAGE,
	      COMMENTS,
	      LINECOUNT,
	      SCHEMA_NAMESPACE,
	      PROP001,
	      PROP002
	    )
	    (SELECT
	        SRC.PROJECT_ID,
	        SRC.CONNECTION_ID,
	        SRC.CATALOG_NAME,
	        SRC.TYPE,
	        :captvId CAPT_VERSION,
	        :verId CONV_VERSION,
	        SRC.SCHEMA_NAME,
	        SRC.PROGRAM_NAME,
	        TAB.ORACLE_NAME,
	        SRC.PROGRAMTYPE,
	        SRC.FUNCTIONTYPE,
	        SRC.NATIVE_SQL,
	        SRC.NATIVE_KEY,
	        SRC.LANGUAGE,
	        SRC.COMMENTS,
	        SRC.LINECOUNT,
	        SRC.SCHEMA_NAMESPACE,
	        SRC.PROP001,
	        SRC.PROP002
	      FROM
	        (SELECT PROJECT_ID,
	          CONNECTION_ID,
	          CATALOG_NAME,
	          'CONVERTED' TYPE,
	          SCHEMA_NAME,
	          PROGRAM_NAME,
	          PROGRAMTYPE,
	          FUNCTIONTYPE,
	          NATIVE_SQL,
	          NATIVE_KEY,
	          LANGUAGE,
	          COMMENTS,
	          LINECOUNT,
	          SCHEMA_NAMESPACE,
	          PROP001,
	          PROP002,
	          ROWID RID
	        FROM MD_STORED_PROGRAMS
	        WHERE PROJECT_ID  = :projectId
	        AND CONNECTION_ID = :svrId
	        AND CAPT_VERSION = :captvId
	        AND TYPE          = 'CAPTURED'
  }';
  
  v_sql1 := q'{
        ) SRC ,
        (SELECT DECODE ( (ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 
           1, UTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
           MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(UTAB.ORACLE_NAME), '_'  || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 128) ) ORACLE_NAME,
          UTAB.RID RID
        FROM
          (SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_TABLES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(VIEW_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_VIEWS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SEQUENCE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SEQUENCES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PROGRAM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_STORED_PROGRAMS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PACKAGE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_PACKAGES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(DATA_TYPE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_USER_DEFINED_DATA_TYPES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SYNONYM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SYNONYMS
          WHERE 
          PRIVATE_VISIBILITY = 'Y'
          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          ) UTAB
        ) TAB
      WHERE SRC.RID = TAB.RID
    )  
  }';
  
  cur := DBMS_SQL.OPEN_CURSOR;
   IF DBMS_LOB.GETLENGTH(v_schemas) > 0  THEN
	 v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	 v_sql := v_sql || v_schemas || v_sql1;
  ELSE
	 v_sql := v_sql || v_sql1;
  END IF;
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);
  DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
  DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
  DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
  DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
  DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
  ex := DBMS_SQL.EXECUTE(cur);
  DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_STORED_PROGRAMS.');
  DBMS_SQL.CLOSE_CURSOR(cur); 
  COMMIT;
  END ConvertStoredPrograms;

  PROCEDURE ConvertSequences(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL)) AS
	v_sql CLOB;
	v_sql1 CLOB;
	v_schemas CLOB;
	cur NUMBER;
	ex NUMBER;
	  
	BEGIN
	v_schemas := convertArrToStr(schemaArr);
	v_sql := q'{
	   INSERT INTO MD_SEQUENCES
		(
		  PROJECT_ID, 
		  CONNECTION_ID,
		  CATALOG_NAME,
		  TYPE,
		  CAPT_VERSION,
	      CONV_VERSION,
		  SCHEMA_NAME,
		  SEQUENCE_NAME,
		  ORACLE_NAME,
		  SEQ_START,
		  INCR,
		  MIN,
		  MAX,
		  SCHEMA_NAMESPACE, 
		  CURRENT_VALUE,
		  COMMENTS
		)
	    (SELECT
	        SRC.PROJECT_ID,
	        SRC.CONNECTION_ID,
	        SRC.CATALOG_NAME,
	        SRC.TYPE,
	        :captvId CAPT_VERSION,
	        :verId CONV_VERSION,
	        SRC.SCHEMA_NAME,
	        SRC.SEQUENCE_NAME,
	        TAB.ORACLE_NAME,
	        SRC.SEQ_START,
	        SRC.INCR,
	        SRC.MIN,
	        SRC.MAX,
	        SRC.SCHEMA_NAMESPACE,
	        SRC.CURRENT_VALUE,
	        SRC.COMMENTS
	      FROM
	        (SELECT 
	          ROWID RID,
	          PROJECT_ID,
	          CONNECTION_ID,
	          CATALOG_NAME,
	          'CONVERTED' TYPE,
	          SCHEMA_NAME,
	          SEQUENCE_NAME,
	          SEQ_START,
			  INCR,
			  MIN,
			  MAX,
			  SCHEMA_NAMESPACE, 
			  CURRENT_VALUE,
			  COMMENTS
	        FROM 
	          MD_SEQUENCES
	        WHERE 
	          PROJECT_ID  = :projectId
	          AND CONNECTION_ID = :svrId
	          AND CAPT_VERSION = :captvId
	          AND TYPE          = 'CAPTURED'
	 }';         
	
   v_sql1 := q'{
        ) SRC,
	        (SELECT 
	          DECODE ( (ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 
	             1, UTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
	             MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(UTAB.ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 128) ) ORACLE_NAME,
	          UTAB.RID RID
	         FROM
	          (SELECT 
	            ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	           FROM 
	            MD_TABLES
	           WHERE 
	            SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	            AND TYPE = 'CAPTURED'
	          UNION
	          SELECT 
	            ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(VIEW_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM 
	            MD_VIEWS
	          WHERE 
	            SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID 
	            AND TYPE = 'CAPTURED'
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SEQUENCE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_SEQUENCES
	          WHERE 
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID 
	          AND TYPE             = 'CAPTURED'
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PROGRAM_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_STORED_PROGRAMS
	          WHERE 
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE = 'CAPTURED'
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PACKAGE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_PACKAGES
	          WHERE
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE             = 'CAPTURED'
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(DATA_TYPE_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_USER_DEFINED_DATA_TYPES
	          WHERE 
	          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE = 'CAPTURED'
	          UNION
	          SELECT ROWID RID,
	            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SYNONYM_NAME) ORACLE_NAME,
	            SCHEMA_NAMESPACE
	          FROM MD_SYNONYMS
	          WHERE 
	          PRIVATE_VISIBILITY = 'Y'
	          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
	          AND TYPE = 'CAPTURED'
	          ) UTAB
	        ) TAB
	      WHERE SRC.RID = TAB.RID
	    )
   }';

   cur := DBMS_SQL.OPEN_CURSOR;
   IF DBMS_LOB.GETLENGTH(v_schemas) > 0  THEN
	 v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	 v_sql := v_sql || v_schemas || v_sql1;
  ELSE
	 v_sql := v_sql || v_sql1;
  END IF;
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);
  DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
  DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
  DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
  DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
  DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
  ex := DBMS_SQL.EXECUTE(cur);
  DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_SEQUENCES.');
  DBMS_SQL.CLOSE_CURSOR(cur);
  COMMIT;
  END ConvertSequences;

  PROCEDURE ConvertTriggers(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB := EMPTY_CLOB();
  v_tables CLOB := EMPTY_CLOB();
  cur NUMBER;
  ex NUMBER;
	  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);

  v_sql := q'{
  INSERT
  INTO MD_TRIGGERS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      SCHEMA_NAME,
      TABLE_OR_VIEW_NAME,
      TRIGGER_NAME,
      ORACLE_NAME,
      TRIGGER_ON_FLAG,
      TRIGGER_TIMING,
      TRIGGER_OPERATION,
      TRIGGER_EVENT,
      NATIVE_SQL,
      NATIVE_KEY,
      LANGUAGE,
      COMMENTS,
      LINECOUNT,
      TRIGGER_NAMESPACE
    )
    (SELECT
        PROJECT_ID,
        CONNECTION_ID,
        CATALOG_NAME,
        TYPE,
        CAPT_VERSION,
        CONV_VERSION, 
        SCHEMA_NAME,
        TABLE_OR_VIEW_NAME,
        TRIGGER_NAME,
        DECODE( (ROW_NUMBER() OVER (PARTITION BY UPPER(ORACLE_NAME), TRIGGER_NAMESPACE ORDER BY TRIGGER_NAME)), 
               1, ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
               MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER( ORACLE_NAME), TRIGGER_NAMESPACE ORDER BY TRIGGER_NAME)), 128) ) ORACLE_NAME,
        TRIGGER_ON_FLAG,
        TRIGGER_TIMING,
        TRIGGER_OPERATION,
        TRIGGER_EVENT,
        NATIVE_SQL,
        NATIVE_KEY,
        LANGUAGE,
        COMMENTS,
        LINECOUNT,
        TRIGGER_NAMESPACE
      FROM
        (SELECT PROJECT_ID,
          CONNECTION_ID,
          CATALOG_NAME,
          'CONVERTED' TYPE,
          CAPT_VERSION,
          :verId CONV_VERSION, 
          SCHEMA_NAME,
          TABLE_OR_VIEW_NAME,
          TRIGGER_NAME,
          MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TRIGGER_NAME) ORACLE_NAME,
          TRIGGER_ON_FLAG,
          TRIGGER_TIMING,
          TRIGGER_OPERATION,
          TRIGGER_EVENT,
          NATIVE_SQL,
          NATIVE_KEY,
          LANGUAGE,
          COMMENTS,
          LINECOUNT,
          TRIGGER_NAMESPACE 
        FROM MD_TRIGGERS
        WHERE PROJECT_ID  = :projectId
        AND CONNECTION_ID = :svrId
        AND CAPT_VERSION = :captvId
        AND TYPE          = 'CAPTURED'
  }';
  
  v_sql1 := q'{
      ) TAB )
  }';

  cur := DBMS_SQL.OPEN_CURSOR;
  IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
      v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas || v_sql1;
  ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
      v_tables := ' AND UPPER(TABLE_OR_VIEW_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables || v_sql1;
  ELSE
	  v_sql := v_sql || v_sql1;
   END IF;



--   IF DBMS_LOB.GETLENGTH(v_schemas) > 0  THEN
--	 v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
--	 v_sql := v_sql || v_schemas || v_sql1;
--  ELSE
--	 v_sql := v_sql || v_sql1;
--  END IF;
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);
  DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
  DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
  DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
  DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
  DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
  ex := DBMS_SQL.EXECUTE(cur);
  DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_TRIGGERS.');
  DBMS_SQL.CLOSE_CURSOR(cur);
  COMMIT;
  END ConvertTriggers;

  PROCEDURE ConvertUDT(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB;
  cur NUMBER;
  ex NUMBER;
	  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_sql := q'{
     INSERT
	  INTO MD_USER_DEFINED_DATA_TYPES
	    (
	      PROJECT_ID,
	      CONNECTION_ID,
	      CATALOG_NAME,
	      TYPE,
	      CAPT_VERSION,
	      CONV_VERSION,
	      SCHEMA_NAME,
	      DATA_TYPE_NAME,
	      ORACLE_NAME,
	      DEFINITION,
	      NATIVE_SQL,
	      NATIVE_KEY,
	      COMMENTS,
	      SCHEMA_NAMESPACE
	    )
	    (SELECT
	        SRC.PROJECT_ID,
	        SRC.CONNECTION_ID,
	        SRC.CATALOG_NAME,
	        SRC.TYPE,
	        :captvId CAPT_VERSION,
	        :verId CONV_VERSION,
	        SRC.SCHEMA_NAME,
	        SRC.DATA_TYPE_NAME,
	        TAB.ORACLE_NAME,
	        SRC.DEFINITION,
	        SRC.NATIVE_SQL,
	        SRC.NATIVE_KEY,
	        SRC.COMMENTS,
	        SRC.SCHEMA_NAMESPACE
	      FROM
	        (SELECT ROWID RID,
	          PROJECT_ID,
	          CONNECTION_ID,
	          CATALOG_NAME,
	          'CONVERTED' TYPE,
	          SCHEMA_NAME,
	          DATA_TYPE_NAME,
	          DEFINITION,
	          NATIVE_SQL,
	          NATIVE_KEY,
	          COMMENTS,
	          SCHEMA_NAMESPACE
	        FROM MD_USER_DEFINED_DATA_TYPES
	        WHERE PROJECT_ID  = :projectId
	        AND CONNECTION_ID = :svrId
	        AND CAPT_VERSION = :captvId
	        AND TYPE          = 'CAPTURED'
  }';

  v_sql1 := q'{
        ) SRC,
        (SELECT DECODE ( (ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 
                   1, UTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
                   MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(UTAB.ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(UTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY UTAB.ORACLE_NAME)), 128) ) ORACLE_NAME,
          UTAB.RID RID
        FROM
          (SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_TABLES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(VIEW_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_VIEWS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SEQUENCE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SEQUENCES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PROGRAM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_STORED_PROGRAMS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PACKAGE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_PACKAGES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(DATA_TYPE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_USER_DEFINED_DATA_TYPES
          WHERE PROJECT_ID     = :projectId
          AND CONNECTION_ID    = :svrId
          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SYNONYM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SYNONYMS
          WHERE 
          PRIVATE_VISIBILITY = 'Y'
          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          ) UTAB
        ) TAB
      WHERE SRC.RID = TAB.RID
    )
  }';
  
  cur := DBMS_SQL.OPEN_CURSOR;
   IF DBMS_LOB.GETLENGTH(v_schemas) > 0  THEN
	 v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	 v_sql := v_sql || v_schemas || v_sql1;
  ELSE
	 v_sql := v_sql || v_sql1;
  END IF;
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);
  DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
  DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
  DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
  DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
  DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
  ex := DBMS_SQL.EXECUTE(cur);
  DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_USER_DEFINED_DATA_TYPES.');
  DBMS_SQL.CLOSE_CURSOR(cur); 
  COMMIT;
  END ConvertUDT;

  PROCEDURE ConvertSynonyms(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_sql1 CLOB;
  v_schemas CLOB;
  cur NUMBER;
  ex NUMBER;
	  
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
  v_sql := q'{
    INSERT
	  INTO MD_SYNONYMS
	    (
	      PROJECT_ID,
	      CONNECTION_ID,
	      CATALOG_NAME,
	      TYPE,
	      CAPT_VERSION,
	      CONV_VERSION,
	      SCHEMA_NAME,
	      SYNONYM_NAME,
	      ORACLE_NAME,
	      SYNONYM_FOR_OBJECT_NAME,
	      FOR_OBJECT_TYPE,
	      PRIVATE_VISIBILITY,
	      NATIVE_SQL,
	      NATIVE_KEY,
	      COMMENTS,
	      SCHEMA_NAMESPACE
	    )
	    (SELECT
	        SRC.PROJECT_ID,
	        SRC.CONNECTION_ID,
	        SRC.CATALOG_NAME,
	        SRC.TYPE,
	        :captvId CAPT_VERSION,
	        :verId CONV_VERSION,
	        SRC.SCHEMA_NAME,
	        SRC.SYNONYM_NAME,
	        TAB.ORACLE_NAME,
	        CASE 
	          WHEN REGEXP_LIKE(SRC.SYNONYM_FOR_OBJECT_NAME, '^(\[?(\w+)\]?)\.?(\[?(\w+)\]?)', 'i') -- SYSDATE
	              THEN REGEXP_REPLACE(SRC.SYNONYM_FOR_OBJECT_NAME, '^(\[?(\w+)\]?)\.?(\[?(\w+)\]?)', '\2.\4', '1', 1, 'i') 
	              ELSE  SRC.SYNONYM_FOR_OBJECT_NAME        
	    	END SYNONYM_FOR_OBJECT_NAME,
	        SRC.FOR_OBJECT_TYPE,
	        SRC.PRIVATE_VISIBILITY,
	        SRC.NATIVE_SQL,
	        SRC.NATIVE_KEY,
	        SRC.COMMENTS,
	        SRC.SCHEMA_NAMESPACE
	      FROM
	        (SELECT ROWID RID,
	          PROJECT_ID,
	          CONNECTION_ID,
	          CATALOG_NAME,
	          'CONVERTED' TYPE,
	          SCHEMA_NAME,
	          SYNONYM_NAME,
	          ORACLE_NAME,
	          SYNONYM_FOR_OBJECT_NAME,
	          FOR_OBJECT_TYPE,
	          PRIVATE_VISIBILITY,
	          NATIVE_SQL,
	          NATIVE_KEY,
	          COMMENTS,
	          SCHEMA_NAMESPACE
	        FROM MD_SYNONYMS
	        WHERE PROJECT_ID  = :projectId
	        AND CONNECTION_ID = :svrId
	        AND CAPT_VERSION = :captvId
	        AND TYPE          = 'CAPTURED'
  }';

 v_sql1 := q'{
       ) SRC,
        (SELECT DECODE ( (ROW_NUMBER() OVER (PARTITION BY UPPER(SYNTAB.ORACLE_NAME), SYNTAB.SCHEMA_NAMESPACE ORDER BY SYNTAB.ORACLE_NAME)), 
           1, SYNTAB.ORACLE_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
           MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(SYNTAB.ORACLE_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(SYNTAB.ORACLE_NAME), SCHEMA_NAMESPACE ORDER BY SYNTAB.ORACLE_NAME)), 128) ) ORACLE_NAME,
          SYNTAB.RID RID
        FROM
          (SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_TABLES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(VIEW_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_VIEWS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SEQUENCE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SEQUENCES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PROGRAM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_STORED_PROGRAMS
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(PACKAGE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_PACKAGES
          WHERE 
          SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(DATA_TYPE_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_USER_DEFINED_DATA_TYPES
          WHERE PROJECT_ID     = :projectId
          AND CONNECTION_ID    = :svrId
          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          UNION
          SELECT ROWID RID,
            MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(SYNONYM_NAME) ORACLE_NAME,
            SCHEMA_NAMESPACE
          FROM MD_SYNONYMS
          WHERE 
          PRIVATE_VISIBILITY = 'Y'
          AND SCHEMA_NAMESPACE = 'NS_SCHEMAOBJ-' || SCHEMA_NAME || '-' || CONNECTION_ID || '-' || PROJECT_ID
          AND TYPE             = 'CAPTURED'
          ) SYNTAB
        ) TAB
      WHERE SRC.RID = TAB.RID
    ) }';

   cur := DBMS_SQL.OPEN_CURSOR;
   IF DBMS_LOB.GETLENGTH(v_schemas) > 0  THEN
	 v_schemas := ' AND UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	 v_sql := v_sql || v_schemas || v_sql1;
   ELSE
	 v_sql := v_sql || v_sql1;
   END IF;
  
   -- DBMS_OUTPUT.PUT_LINE(v_sql);
   DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
   DBMS_SQL.BIND_VARIABLE(cur, 'captvId', v_captver);
   DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_versionId);
   DBMS_SQL.BIND_VARIABLE(cur, 'svrId', nConnId);
   DBMS_SQL.BIND_VARIABLE(cur, 'projectId', nProjectId);
   ex := DBMS_SQL.EXECUTE(cur);
   DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_SYNONYMS.');
   DBMS_SQL.CLOSE_CURSOR(cur); 
  COMMIT;
  END ConvertSynonyms;

 PROCEDURE PostProcessing(nProjectId IN NUMBER, nConnId IN NUMBER, v_srcdatabase IN VARCHAR2,
     schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL))  AS
 v_schemas CLOB := EMPTY_CLOB();
 v_tables CLOB := EMPTY_CLOB();
 v_sql CLOB := EMPTY_CLOB();
 v_cnt NUMBER;
 v_schcnt NUMBER;
 v_tblcnt NUMBER;
   
 BEGIN
 
  IF DBMS_DB_VERSION.VERSION < 12 THEN
    -- Emulate Autoinc./Identity Columns
    INSERT INTO MD_SEQUENCES
    (
      PROJECT_ID, 
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      SCHEMA_NAME,
      SEQUENCE_NAME,
      ORACLE_NAME,
      SEQ_START,
      INCR,
      SCHEMA_NAMESPACE, 
      CURRENT_VALUE,
      COMMENTS
    )
    (
        SELECT
          PROJECT_ID, 
	      CONNECTION_ID,
	      CATALOG_NAME,
	      TYPE,
          CAPT_VERSION,
          CONV_VERSION,
	      SCHEMA_NAME,
	      SEQUENCE_NAME,
	      ORACLE_NAME,
	      SEQ_START,
	      INCR,
	      SCHEMA_NAMESPACE, 
          CURRENT_VALUE,
          COMMENTS
	     FROM
	       (
				 SELECT
					PROJECT_ID, 
					CONNECTION_ID,
					CATALOG_NAME,
					TYPE,
                    CAPT_VERSION,
                    CONV_VERSION,
					SCHEMA_NAME,
					MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME || '_' || COLUMN_NAME || '_SEQ') SEQUENCE_NAME,
					MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME || '_' || COLUMN_NAME || '_SEQ') ORACLE_NAME,
					SEQ_START,
					INCR,
					SCHEMA_NAMESPACE,
                    CURRENT_VALUE,
                    COMMENTS
				  FROM 
					   (
					     SELECT 
					       C.PROJECT_ID, 
					       C.CONNECTION_ID,
					       C.CATALOG_NAME,
					       C.TYPE,
                           CAPT_VERSION,
                           v_versionId CONV_VERSION,
					       C.SCHEMA_NAME,
					       (SELECT ORACLE_NAME 
					         FROM MD_TABLES 
					         WHERE PROJECT_ID = C.PROJECT_ID AND CONNECTION_ID = C.CONNECTION_ID AND CATALOG_NAME = C.CATALOG_NAME
					         AND TYPE = C.TYPE AND SCHEMA_NAME = C.SCHEMA_NAME AND TABLE_NAME = C.TABLE_NAME) TABLE_NAME, 
					       C.ORACLE_NAME COLUMN_NAME,
					       C.AUTOINC_SEEDVALUE SEQ_START,
					       C.AUTOINC_INCREMENTVALUE INCR,
					       (SELECT SCHEMA_NAMESPACE 
					         FROM MD_TABLES 
					         WHERE PROJECT_ID = C.PROJECT_ID AND CONNECTION_ID = C.CONNECTION_ID AND CATALOG_NAME = C.CATALOG_NAME
					         AND TYPE = C.TYPE AND SCHEMA_NAME = C.SCHEMA_NAME AND TABLE_NAME = C.TABLE_NAME) SCHEMA_NAMESPACE,
                            C.AUTOINC_SEEDVALUE CURRENT_VALUE,
                            C.COMMENTS
					     FROM
					       MD_COLUMNS C
					     WHERE
						   C.ISAUTOINC = 'Y'
						   AND C.TYPE = 'CONVERTED'
						   AND C.PROJECT_ID = nProjectId
				           AND C.CONNECTION_ID = nConnId
                           AND C.CAPT_VERSION = v_captver
					   )    
             )
       
    );
    
    DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_SEQUENCES.');
    COMMIT;
    
    
    INSERT INTO MD_TRIGGERS
    (
      PROJECT_ID, 
      CONNECTION_ID,
      CATALOG_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      SCHEMA_NAME,
      TABLE_OR_VIEW_NAME,
      TRIGGER_NAME,
      ORACLE_NAME,
      NATIVE_SQL,
      LANGUAGE,
      TRIGGER_NAMESPACE,
      TRIGGER_ON_FLAG,
      COMMENTS
    )
    (
        SELECT
          TAB.PROJECT_ID, 
          TAB.CONNECTION_ID,
          TAB.CATALOG_NAME,
          TAB.TYPE,
          TAB.CAPT_VERSION,
          TAB.CONV_VERSION,
          TAB.SCHEMA_NAME,
          TAB.TABLE_NAME,
          TAB.TRIGGER_NAME,
          TAB.ORACLE_NAME,
          -- Better way to get a Sequence name is to query MD_SEQUENCES.
          MIGRATION_TRANSFORMER.GetIdentityTrigger(TAB.ORACLE_NAME, TAB.TABLE_NAME, TAB.COLUMN_NAME, TAB.TABLE_NAME || '_' || TAB.COLUMN_NAME || '_SEQ') NATIVE_SQL,
          'OracleSQL' LANGUAGE,
          'NS_DB_TRIGGERS-' || SCHEMA_NAME || '-' || nConnId || '-' || nProjectId TRIGGER_NAMESPACE,
          'T' TRIGGER_ON_FLAG,
          COMMENTS
	     FROM
	       (
				 SELECT
					PROJECT_ID, 
					CONNECTION_ID,
					CATALOG_NAME,
					TYPE,
 					CAPT_VERSION,
      				CONV_VERSION,
					SCHEMA_NAME,
					TABLE_NAME,
					COLUMN_NAME,
					MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME || '_' || COLUMN_NAME || '_TRG') TRIGGER_NAME,
					MIGRATION_TRANSFORMER.TRANSFORM_IDENTIFIER(TABLE_NAME || '_' || COLUMN_NAME || '_TRG') ORACLE_NAME,
					COMMENTS
				  FROM 
				   (
				     SELECT 
					   C.PROJECT_ID, 
					   C.CONNECTION_ID,
					   C.CATALOG_NAME,
					   C.TYPE,
					   C.CAPT_VERSION,
					   v_versionId CONV_VERSION,
					   C.SCHEMA_NAME,
					   (SELECT ORACLE_NAME 
					     FROM MD_TABLES 
					     WHERE PROJECT_ID = C.PROJECT_ID AND CONNECTION_ID = C.CONNECTION_ID AND CATALOG_NAME = C.CATALOG_NAME
					     AND TYPE = C.TYPE AND SCHEMA_NAME = C.SCHEMA_NAME AND TABLE_NAME = C.TABLE_NAME) TABLE_NAME, 
					   C.ORACLE_NAME COLUMN_NAME,
					   C.COMMENTS
				     FROM
				       MD_COLUMNS C
				     WHERE
				       C.ISAUTOINC = 'Y'
				       AND C.TYPE = 'CONVERTED'
				       AND C.PROJECT_ID = nProjectId
				       AND C.CONNECTION_ID = nConnId
                       AND C.CAPT_VERSION = v_captver
				   )
	              
	       ) TAB   
       
    );
    
    DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_TRIGGERS.');
    COMMIT;
  END IF;
  
  
  -- Update the Captured Version(v_captver) rows by updating CONV_VERSION with v_versionId so that it is in sync with the actual CONV_VERSION(that has this v_versionId) 
  -- in Converted Version rows for the Project(i.e., ProjectId) that we are working on. If we are filtering then we DON'T update as long as the filtered schemas 
  -- and tables are less than their actual Captured Version.
 
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);

  SELECT COUNT(CONSTRAINT_NAME) INTO v_cnt  FROM USER_CONSTRAINTS WHERE TABLE_NAME LIKE 'MD_%' AND CONSTRAINT_NAME LIKE 'MD_%';
  -- DBMS_OUTPUT.PUT_LINE('Number of Constraints in the Model: ' || v_cnt);
  SELECT COUNT(SCHEMA_NAME) INTO v_schcnt FROM MD_SCHEMAS WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND TYPE = 'CAPTURED' AND CAPT_VERSION = v_captver; 
  -- DBMS_OUTPUT.PUT_LINE('Number of Schemas for Captured Version ' || v_captver || ': ' || v_schcnt) ;
  SELECT COUNT(TABLE_NAME) INTO v_tblcnt FROM MD_TABLES WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND TYPE = 'CAPTURED' AND CAPT_VERSION = v_captver;
  -- DBMS_OUTPUT.PUT_LINE('Number of Tables for Captured Version ' ||v_captver ||': ' || v_tblcnt) ;
  
  IF v_cnt > 0 THEN
	   -- Disable all the constraints 
	   FOR I IN (SELECT CONSTRAINT_NAME, TABLE_NAME FROM USER_CONSTRAINTS WHERE TABLE_NAME LIKE 'MD_%' AND CONSTRAINT_NAME LIKE 'MD_%') 
	   LOOP
	      EXECUTE IMMEDIATE 'ALTER TABLE ' || I.TABLE_NAME || ' DISABLE CONSTRAINT ' || I.CONSTRAINT_NAME || ' CASCADE'  || '';
	   END LOOP;
	   COMMIT;
  END IF;
  
--  DBMS_OUTPUT.PUT_LINE('Schema Array Count: ' || schemaArr.COUNT);
--  DBMS_OUTPUT.PUT_LINE('Schema Array First: ' || schemaArr(schemaArr.FIRST));
--  DBMS_OUTPUT.PUT_LINE('Table Array Count: ' || tableArr.COUNT);
--  DBMS_OUTPUT.PUT_LINE('Table Array First: ' || tableArr(tableArr.FIRST));

  IF (schemaArr(schemaArr.FIRST) IS NULL AND tableArr(tableArr.FIRST) IS NULL) OR
      (schemaArr(schemaArr.FIRST) IS NOT NULL AND v_schcnt = schemaArr.COUNT AND tableArr(tableArr.FIRST) IS NULL) OR 
      (schemaArr(schemaArr.FIRST) IS NOT NULL AND v_schcnt = schemaArr.COUNT AND tableArr(tableArr.FIRST) IS NOT NULL AND v_tblcnt = tableArr.COUNT) THEN
       
      -- DBMS_OUTPUT.PUT_LINE('Do updates for Captured Version');
      UPDATE MD_CATALOGS 
      SET CONV_VERSION = v_versionId 
      WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_SCHEMAS 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_TABLES 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_COLUMNS 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_CONSTRAINTS 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_INDEXES 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_TRIGGERS 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';
     
      UPDATE MD_SYNONYMS 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_SEQUENCES 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_USER_DEFINED_DATA_TYPES 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';

      UPDATE MD_STORED_PROGRAMS 
	  SET CONV_VERSION = v_versionId 
	  WHERE PROJECT_ID = nProjectId AND CONNECTION_ID = nConnId AND CAPT_VERSION = v_captver AND TYPE = 'CAPTURED';   

  END IF;
  
  IF v_cnt > 0 THEN
	   -- Enable PK's and CHECK's 
	   FOR I IN (SELECT CONSTRAINT_NAME, TABLE_NAME FROM USER_CONSTRAINTS 
	       WHERE TABLE_NAME LIKE 'MD_%' AND CONSTRAINT_NAME LIKE 'MD_%' 
	       AND (CONSTRAINT_TYPE = 'P' OR CONSTRAINT_TYPE = 'C')) 
	   LOOP
	      EXECUTE IMMEDIATE 'ALTER TABLE ' || I.TABLE_NAME || ' ENABLE CONSTRAINT ' || I.CONSTRAINT_NAME ||'';
	   END LOOP;
	   
	   -- Enable FK's
	   FOR I IN (SELECT CONSTRAINT_NAME, TABLE_NAME FROM USER_CONSTRAINTS 
	       WHERE TABLE_NAME LIKE 'MD_%' AND CONSTRAINT_NAME LIKE 'MD_%' 
	        AND (CONSTRAINT_TYPE = 'R')) 
	   LOOP
	      EXECUTE IMMEDIATE 'ALTER TABLE ' || I.TABLE_NAME || ' ENABLE CONSTRAINT ' || I.CONSTRAINT_NAME ||'';
	   END LOOP; 
  END IF;
  COMMIT;
   
 END PostProcessing;
 
 PROCEDURE CONVERT(projectId IN NUMBER, connId IN NUMBER, srcDb IN VARCHAR2, 
           schemaArr IN MIGRATION_CONVERT.NAMES, tableArr IN MIGRATION_CONVERT.NAMES, capVer IN NUMBER DEFAULT 0) AS
   BEGIN
      -- DBMS_OUTPUT.PUT_LINE('Convert Version: ' || v_capver);
      PreProcessing(projectId, connId, srcDb, capVer);
      ConvertDatabases(projectId, connId, srcDb, schemaArr, tableArr);
      ConvertSchemas(projectId, connId, srcDb, schemaArr, tableArr);
      ConvertTables(projectId, connId, srcDb, schemaArr, tableArr);
      ConvertUDT(projectId, connId, srcDb, schemaArr);
      ConvertColumns(projectId, connId, srcDb, schemaArr, tableArr);
      ConvertConstraints(projectId, connId, srcDb, schemaArr, tableArr);
      ConvertIndexes(projectId, connId, srcDb, schemaArr, tableArr);
      ConvertViews(projectId, connId, srcDb, schemaArr);
      ConvertStoredPrograms(projectId, connId, srcDb, schemaArr);
      ConvertTriggers(projectId, connId, srcDb,schemaArr, tableArr);
      ConvertSynonyms(projectId, connId, srcDb, schemaArr);
      ConvertSequences(projectId, connId, srcDb, schemaArr);
      PostProcessing(projectId, connId, srcDb, schemaArr, tableArr);
  EXCEPTION
    WHEN missing_ver THEN
       DBMS_OUTPUT.PUT_LINE ('Captured Version for Database: ' || srcDb || ', Captured Version: ' || capVer || ', Project Id: ' || projectId || 
           ' and Server Id: ' || connId || q'[ doesn't exist. Unable to do the Conversion.]' );
  END CONVERT;

  PROCEDURE CONVERT(projectId IN NUMBER, connId IN NUMBER, srcDb IN VARCHAR2, 
           schemaJArr IN VARCHAR2, tableJArr IN VARCHAR2, capVer IN NUMBER DEFAULT 0) AS
  v_schemaArr MIGRATION_CONVERT.NAMES := MIGRATION_CONVERT.NAMES(NULL);
  v_tableArr MIGRATION_CONVERT.NAMES := MIGRATION_CONVERT.NAMES(NULL);
  BEGIN
	  v_schemaArr := convertJSONArrStrToNames(schemaJArr);
	  v_tableArr := convertJSONArrStrToNames(tableJArr);
	  CONVERT(projectId=>projectId, connId=>connId, srcDb=>srcDb, schemaArr=>v_schemaArr, tableArr=>v_tableArr, capVer=>capVer);
  EXCEPTION
    WHEN OTHERS THEN
    RAISE;
  END CONVERT;
  
  PROCEDURE CONVERT(projectId IN NUMBER, connId IN NUMBER, srcDb IN VARCHAR2, 
    capVer IN NUMBER DEFAULT 0) AS
    v_schemaArr MIGRATION_CONVERT.NAMES := MIGRATION_CONVERT.NAMES(NULL);
    v_tableArr MIGRATION_CONVERT.NAMES := MIGRATION_CONVERT.NAMES(NULL);
    BEGIN
	  CONVERT(projectId=>projectId, connId=>connId, srcDb=>srcDb, schemaArr=>v_schemaArr, tableArr=>v_tableArr, capVer=>capVer);
    EXCEPTION
      WHEN OTHERS THEN
      RAISE;
  END CONVERT;
  

END MIGRATION_CONVERT;
/
           