
CREATE OR REPLACE TYPE bridgeCommandsArray AS VARRAY(10) OF VARCHAR2(32000);
/

--CREATE OR REPLACE TYPE NAMES AS TABLE OF VARCHAR2(128);
--/

CREATE OR REPLACE PACKAGE "REDSHIFT_CAPTURE" AS

PROCEDURE deleteCapturedMetadata(projectName Varchar2);

FUNCTION getOnlineRedshiftBridgeCommands RETURN bridgeCommandsArray;



-- Capture Code from REDSHIFT_CAPTURE_HEADER 

  TYPE NAMES IS TABLE OF VARCHAR2(128); 

  FUNCTION CAPTURE(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2, 
           schemaArr IN REDSHIFT_CAPTURE.NAMES, tableArr IN REDSHIFT_CAPTURE.NAMES DEFAULT NAMES(NULL)) RETURN VARCHAR2;
           
  FUNCTION CAPTURE(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2, 
           schemaJArr IN VARCHAR2, tableJArr IN VARCHAR2 DEFAULT '[]') RETURN VARCHAR2;
  
  FUNCTION CAPTURE(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2 ) RETURN VARCHAR2;


/*
PROCEDURE dropRepository;
PROCEDURE truncateRepository;

FUNCTION getAllProjectNames return varcharArray;
FUNCTION getProjectDescription(projectName IN VARCHAR2) return VARCHAR2;
FUNCTION getAllCaptureVersions(projectName IN VARCHAR2) return numberArray;
FUNCTION getAllConvertVersions(projectName IN VARCHAR2, captureVersion in NUMBER) return numberArray;
FUNCTION getAllGenerateVersions(projectName IN VARCHAR2, captureVersion in NUMBER, convertVersion in NUMBER) return numberArray;
*/

END;
/


CREATE OR REPLACE PACKAGE BODY "REDSHIFT_CAPTURE" AS

  v_captver NUMBER := 0;
  v_conver NUMBER := 0; -- ALWAYS 0

FUNCTION getProjectID(projectName Varchar2) RETURN number
IS
projectId number;
BEGIN

	select project_id into projectId from MD_PROJECTS where project_name = projectName;
    return projectId;
    
END getProjectID;


FUNCTION getOnlineRedshiftBridgeCommands RETURN bridgeCommandsArray
IS

bridgeCommands bridgeCommandsArray;
BEGIN

	DBMS_OUTPUT.PUT_LINE(' IN getOnlineDatabaseBridgeCommand');
	bridgeCommands := bridgeCommandsArray();
	
	bridgeCommands.EXTEND;
	bridgeCommands(1) := q'[
			BRIDGE MD_STAGE_PG8_DATABASES AS <REDSHIFT_JDBC_CONNECTION>(
			select distinct
			 inet_server_addr() server_address,
			 inet_server_port() server_port,
			 cast(table_catalog as varchar(1000)) as catalog_name,
             cast(table_schema as varchar(1000)) as schema_name,
             cast(current_user as varchar(1000)) as schema_owner
			from 
			 information_schema.tables 
			where 
			 table_schema NOT IN ('pg_internal', 'information_schema', 'pg_catalog')
			 and table_schema in (<SCHEMA_ARRAY>)
			)APPEND
		]';
		
	bridgeCommands.EXTEND;	
	
	bridgeCommands(2) := q'[
			BRIDGE MD_STAGE_PG8_TABLES AS <REDSHIFT_JDBC_CONNECTION>(
			select distinct
               cast (  q1.catalog_name as varchar (1000)) as catalog_name, 
               cast (  q1.schema_name as varchar (1000)) as schema_name, 
               cast (  q1.table_name as varchar (1000)) as table_name, 
               cast (  q2.num_rows as bigint) as num_rows_in_table
            from 
               ( select relname table_name, cast (reltuples as bigint) num_rows from pg_class c
                 JOIN pg_catalog.pg_namespace n ON n.oid = c.relnamespace
                 where relkind in ('r', 'v') and nspname  in (<SCHEMA_ARRAY>)) q2,         
               (select 
	              cast ( table_catalog as varchar (1000))  as catalog_name ,
                  cast ( table_schema  as varchar (1000))  as schema_name ,
                  cast ( table_name    as varchar (1000))  as table_name
                from 
	               information_schema.tables 
                where 
	               table_schema NOT IN ('pg_internal', 'information_schema', 'pg_catalog') and table_type = 'BASE TABLE' 
	               and table_schema in (<SCHEMA_ARRAY>)
                order by 
                   table_name) q1
            where
                 q1.table_name = q2.table_name
			 )APPEND
		]';
	
	
	bridgeCommands.EXTEND;	
	bridgeCommands(3) := q'[
			BRIDGE MD_STAGE_PG8_COLUMNS AS <REDSHIFT_JDBC_CONNECTION>(
			SELECT 
			    cast ( current_database()::information_schema.sql_identifier AS varchar (1000)) AS catalog_name,
			    cast ( nc.nspname::information_schema.sql_identifier AS varchar (1000))  AS schema_name,
			    cast ( c.relname::information_schema.sql_identifier AS varchar (1000)) AS table_name,
			    cast ( a.attname::information_schema.sql_identifier AS varchar (1000)) AS column_name,
			    a.attnum::information_schema.cardinal_number AS ordinal_position,
			    cast ( pg_get_expr(ad.adbin, ad.adrelid)::information_schema.character_data AS varchar(1000)) AS column_default,
			    cast ( CASE
			        WHEN t.typtype = 'd'::"char" THEN
			        CASE
			            WHEN bt.typelem <> 0::oid AND bt.typlen = CAST('-1' AS Integer) THEN format_type(a.atttypid, a.atttypmod)
			            WHEN nbt.nspname = CAST('pg_catalog' AS name) THEN format_type(t.typbasetype, CAST(NULL AS Integer))
			            ELSE format_type(a.atttypid, a.atttypmod)
			        END
			        ELSE
			        CASE
			            WHEN t.typelem <> 0::oid AND  t.typlen = CAST('-1' AS Integer) THEN format_type(a.atttypid, a.atttypmod)
			            WHEN nt.nspname = CAST('pg_catalog' AS name) THEN format_type(a.atttypid, CAST(NULL AS Integer))
			            ELSE format_type(a.atttypid, a.atttypmod)
			        END
			    END::information_schema.character_data AS varchar (1000)) AS data_type,
			    information_schema._pg_char_max_length(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))::information_schema.cardinal_number AS character_maximum_length,
			    information_schema._pg_char_octet_length(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))::information_schema.cardinal_number AS character_octet_length,
			    information_schema._pg_numeric_precision(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))::information_schema.cardinal_number AS numeric_precision,
			    information_schema._pg_numeric_precision_radix(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))::information_schema.cardinal_number AS numeric_precision_radix,
			    information_schema._pg_numeric_scale(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))::information_schema.cardinal_number AS numeric_scale,
			    information_schema._pg_datetime_precision(information_schema._pg_truetypid(a.*, t.*), information_schema._pg_truetypmod(a.*, t.*))::information_schema.cardinal_number AS datetime_precision,
			    cast ( NULL::integer::information_schema.cardinal_number AS varchar (1000)) AS interval_precision,
			    cast ( NULL::character varying::information_schema.sql_identifier AS varchar (1000)) AS character_set_catalog,
			    cast ( NULL::character varying::information_schema.sql_identifier AS varchar (1000)) AS character_set_schema,
			    cast ( NULL::character varying::information_schema.sql_identifier AS varchar (1000)) AS character_set_name,
			    cast ( CASE
			        WHEN t.typtype = 'd'::"char" THEN current_database()
			        ELSE CAST( NULL AS name)
			    END::information_schema.sql_identifier AS varchar (1000)) AS domain_catalog,
			    cast ( CASE
			        WHEN t.typtype = 'd'::"char" THEN nt.nspname
			        ELSE CAST( NULL AS name)
			    END::information_schema.sql_identifier AS varchar (1000)) AS domain_schema,
			    cast ( CASE
			        WHEN t.typtype = 'd'::"char" THEN t.typname
			        ELSE CAST( NULL AS name)
			    END::information_schema.sql_identifier AS varchar (1000)) AS domain_name,
			    cast ( current_database()::information_schema.sql_identifier AS varchar (1000)) AS udt_catalog,
			    cast ( COALESCE(nbt.nspname, nt.nspname)::information_schema.sql_identifier AS varchar (1000)) AS udt_schema,
			    cast ( COALESCE(bt.typname, t.typname)::information_schema.sql_identifier AS varchar (1000)) AS udt_name,
			    cast ( NULL::character varying::information_schema.sql_identifier AS varchar (1000)) AS scope_catalog,
			    cast ( NULL::character varying::information_schema.sql_identifier AS varchar (1000)) AS scope_schema,
			    cast ( NULL::character varying::information_schema.sql_identifier AS varchar (1000)) AS scope_name,
			    NULL::integer::information_schema.cardinal_number AS maximum_cardinality,
			    cast ( a.attnum::information_schema.sql_identifier AS varchar (1000)) AS dtd_identifier,
			    cast ( NULL::character varying::information_schema.character_data AS varchar (1000)) AS identity_generation,
			    cast ( NULL::character varying::information_schema.character_data AS varchar (1000)) AS identity_start,
			    cast ( NULL::character varying::information_schema.character_data AS varchar (1000)) AS identity_increment,
			    cast ( NULL::character varying::information_schema.character_data AS varchar (1000)) AS identity_maximum,
			    cast ( NULL::character varying::information_schema.character_data AS varchar (1000)) AS identity_minimum,
			    cast ( CAST ('NEVER' AS character varying)::information_schema.character_data AS varchar (1000)) AS is_generated,
			    cast ( NULL::character varying::information_schema.character_data AS varchar (1000)) AS generation_expression
			FROM pg_attribute a
			     LEFT JOIN pg_attrdef ad ON a.attrelid = ad.adrelid AND a.attnum = ad.adnum
			     JOIN (pg_class c
			     JOIN pg_namespace nc ON c.relnamespace = nc.oid) ON a.attrelid = c.oid
			     JOIN (pg_type t
			     JOIN pg_namespace nt ON t.typnamespace = nt.oid) ON a.atttypid = t.oid
			     LEFT JOIN (pg_type bt
			     JOIN pg_namespace nbt ON bt.typnamespace = nbt.oid) ON t.typtype = 'd'::"char" AND t.typbasetype = bt.oid
			WHERE 
			   a.attnum > 0 AND NOT a.attisdropped AND (c.relkind = ANY (ARRAY['r'::"char", 'v'::"char", 'f'::"char"]))  
			   AND nc.nspname NOT IN ('pg_catalog', 'information_schema') AND c.relkind = 'r' 
			   and nc.nspname in (<SCHEMA_ARRAY>)
			ORDER BY 
			  table_name, ordinal_position
			)APPEND
		]';
	
	
	bridgeCommands.EXTEND;	
	bridgeCommands(4) := q'[
		BRIDGE MD_STAGE_PG8_COMMENTS AS <REDSHIFT_JDBC_CONNECTION>(
		       select 
		          cast( objects_with_comments.catalog_name as varchar(1000)), 
		          objects_with_comments.schema_name::text, 
		          objects_with_comments.object_name::text, 
		          objects_with_comments.parent_name::text, 
		          objects_with_comments.object_type::text, 
		          objects_with_comments.comments_text::text 
		       from 
		       (
		          select 
		             current_database()::information_schema.sql_identifier AS catalog_name, 
		             ns3.table_schema       schema_name, 
		             ns3.relation_name      object_name, 
		             ns3.parent_name        parent_name, 
		             case ns3.relation_kind  
		                when 'r' then 'table' 
		                when 'v' then 'view' 
		                when 'S' then 'sequence' 
		                when 'i' then 'index' 
		                when 'm' then 'materialized view' 
		                when 'c' then 'composite type' 
		                when 'f' then 'foreign table'  
                        when 'col' then 'column'
		            end as                 object_type, 
		            pgd.description        comments_text 
		          from ( 
		             select 
		               ns2.nsp1 table_schema, ns2.parent_name parent_name, pga.attrelid parent_relation_id, pga.attname relation_name, 
		               'col' relation_kind, pga.attnum attnum 
		             from pg_attribute pga, 
		             ( select 
		                  ns1.nsp nsp1, pgc.oid oid, pgc.relname parent_name, pgc.relnamespace, pgc.reltype reltype, pgc.relkind relkind 
		               from pg_class pgc 
		               join ( 
		                   select nspname nsp, oid oid from pg_namespace 
		                   where nspname not in ('pg_catalog', 'information_schema', 'pg_toast', 'pg_toast_temp_1')
		                   and nspname in (<SCHEMA_ARRAY>) ) ns1 
		               on ns1.oid = pgc.relnamespace and pgc.relkind not in ('i') 
		             ) ns2  
		             where pga.attrelid = ns2.oid and attnum > 0 ) ns3 
		          left join pg_description pgd  
		          on pgd.objoid = ns3.parent_relation_id  and pgd.objsubid = ns3.attnum 
		          union 
		          select 
		             current_database()::information_schema.sql_identifier AS catalog_name, 
		             ns4.schema_name schema_name, 
		             ns4.table_name  object_name, 
		             NULL            parent_name,  
		             'table'         object_type, 
		             pgdesc.description comments_text 
		         from 
		         ( 
		             select 
		                ns3.schema_name schema_name, ns3.ns_oid ns_oid, ns3.table_name table_name, ns3.table_oid table_oid, 
		                (select pg_class.oid from pg_class where relname = 'pg_tables') table_class_oid, 
		                ns3.pg_class_id pg_class_id, 0 only_table_flag 
		             from 
		             ( 
		                select ns1.nsp schema_name, pgc.relnamespace ns_oid, pgc.oid table_oid, pgc.relname table_name, 
		                       (select pg_class.oid from pg_class where relname = 'pg_class') pg_class_id 
		                from pg_class pgc 
		                join ( select nspname nsp, oid 
		                       from pg_namespace 
		                       where nspname not in ('pg_catalog', 'information_schema', 'pg_toast', 'pg_toast_temp_1') 
		                       and nspname in (<SCHEMA_ARRAY>)
		                ) ns1 
		                on ns1.oid = pgc.relnamespace and relname in (select tablename from pg_tables) 
		             ) ns3 
		             join pg_depend pgd 
		             on pgd.objid = ns3.table_oid and pgd.refobjid = ns3.ns_oid 
		         ) ns4 
		         left join pg_description pgdesc 
		         on pgdesc.objoid = ns4.table_oid and pgdesc.classoid = ns4.pg_class_id and pgdesc.objsubid = only_table_flag 
		       ) 
		       objects_with_comments
		       where comments_text is not null
		)APPEND
	]';
	
	
/* No Constraints

	bridgeCommands.EXTEND;	
	bridgeCommands(5) := q'[
			BRIDGE MD_STAGE_PG8_CONSTRAINTS AS <REDSHIFT_JDBC_CONNECTION>(
			SELECT 
			    current_database() AS catalog_name,
				TRIM(cn.nspname) AS schema_name,
				TRIM(c.conname)  AS constraint_name,
				DECODE(c.contype, 
			          'c', 'CHECK',
					  'p', 'PRIMARY KEY',
					  'u', 'UNIQUE',
					  'f', 'FOREIGN KEY'
				) AS constraint_type,
				TRIM(cc.relname) AS table_name,
				TRIM(ca.attname) AS column_name,
				c_ord AS ordinal_position,
				current_database() AS refcatalog_name,
			   	TRIM(fn.nspname) AS refschema_name,
				TRIM(fc.relname) AS reftable_name,
				TRIM(fa.attname) AS refcolumn_name,
			    pg_get_constraintdef(c.oid) as constraint_clause 
			FROM 
			     pg_constraint c
			     LEFT JOIN pg_attribute ca 
			         ON c.conrelid=ca.attrelid  AND ca.attnum=ANY(c.conkey)
			     LEFT JOIN pg_attribute fa 
			         ON c.confrelid=fa.attrelid AND fa.attnum=ANY(c.confkey)
			     LEFT JOIN pg_class cc 
			         ON ca.attrelid=cc.oid
			     LEFT JOIN pg_class fc 
			         ON c.confrelid=fc.oid
			     LEFT JOIN generate_series(1,1600) as tpos(c_ord) 
			         ON ca.attnum=c.conkey[tpos.c_ord]
			     LEFT JOIN generate_series(1,1600) as fpos(f_ord) 
			         ON fa.attnum=c.confkey[fpos.f_ord]
			     LEFT JOIN pg_namespace cn 
			         ON c.connamespace=cn.oid
			     LEFT JOIN pg_namespace fn 
			         ON fc.relnamespace=fn.oid
			 WHERE 
			       c.contype IN ('c','p','u','f') AND cc.relkind = 'r'
				   AND NOT ca.attisdropped
			 ORDER BY 
			      schema_name,
				  constraint_name,
				  c_ord,
				  DECODE(c.contype,'c',0,'p',1,'u',2,'f',3),
					ca.attnum,
					f_ord
			 )APPEND
		]';
*/
		
/*  We do not need Indexes for now 

	bridgeCommands.EXTEND;	
	bridgeCommands(6) := q'[
		BRIDGE MD_STAGE_PG8_INDEXES AS <REDSHIFT_JDBC_CONNECTION>(
		SELECT
		  current_database() AS catalog_name,
		  current_schema() as schema_name,
		  t.relname as table_name,
		  i.relname as index_name,
		  a.attname as column_name
		FROM
		  pg_class t,
		  pg_class i,
		  pg_index ix,
		  pg_attribute a
		WHERE
		  t.oid = ix.indrelid
		  AND i.oid = ix.indexrelid
		  AND a.attrelid = t.oid
		  AND POSITION(a.attname in pg_get_indexdef(ix.indexrelid)) > 0
		  AND t.relkind = 'r'
		  AND t.relname !~ '^pg_'
		  AND ix.indisunique = false 
		  AND ix.indisprimary = false
		ORDER BY
		  t.relname,
		  i.relname)APPEND	
		]';

*/
		DBMS_OUTPUT.PUT_LINE(' OUT getOnlineDatabaseBridgeCommand');
	
	return bridgeCommands;
	
END getOnlineRedshiftBridgeCommands;

PROCEDURE deleteCapturedMetadata(projectName Varchar2) 
IS
projectId number;
BEGIN
	
	projectId := getProjectID(projectName);

	DBMS_OUTPUT.PUT_LINE(' Project ID  ' || projectId);
	
	DELETE FROM MD_GENERATE WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_DATATYPE_TRANSFORM_RULE WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_DATATYPE_TRANSFORM_MAP WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_CONSTRAINTS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_INDEXES WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_COLUMNS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_TRIGGERS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_STORED_PROGRAMS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_TABLES WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_VIEWS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_SCHEMAS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_CATALOGS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_CONNECTIONS WHERE PROJECT_ID_FK = projectId ;
	DELETE FROM MD_PROJECTS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_STAGE_CONSTRAINTS WHERE PROJECT_ID = projectId ;
	DELETE FROM MD_STAGE_INDEXES WHERE PROJECT_ID = projectId ;
COMMIT;

END deleteCapturedMetadata;


-- Code from MIGRATION_CAPTURE_BODY

  FUNCTION convertArrToStr(namesArr IN NAMES) RETURN CLOB AS
  v_names CLOB := EMPTY_CLOB();
  BEGIN
	  IF namesArr IS NOT NULL THEN
	     FOR i in 1 .. namesArr.COUNT LOOP
	      IF namesArr(i) IS NOT NULL THEN
		      IF i = 1 THEN 
		         v_names := q'[']' || UPPER(namesArr(i)) || q'[']';
		      ELSE
		         v_names := v_names || ', ' || q'[']' || UPPER(namesArr(i)) || q'[']';
		      END IF;
		  END IF;   
	    END LOOP;
   END IF;
   -- DBMS_OUTPUT.PUT_LINE('LIST: ' || v_names); 
   RETURN v_names;
  END convertArrToStr;
  
  FUNCTION convertJSONArrStrToNames(jsonArr IN VARCHAR2) RETURN REDSHIFT_CAPTURE.NAMES AS
  v_namesType REDSHIFT_CAPTURE.NAMES;
  BEGIN
	  IF jsonArr IS NOT NULL AND LENGTH(jsonArr) > 0 THEN
		SELECT 
		   VALUE BULK COLLECT into v_namesType
		FROM JSON_TABLE( jsonArr, '$[*]'	COLUMNS (VALUE PATH '$'));
	 END IF;
	 RETURN v_namesType;
  END convertJSONArrStrToNames;

  PROCEDURE CaptureConnections(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_cnt NUMBER;
  BEGIN
	  
	SELECT COUNT(PROJECT_ID) INTO v_cnt FROM MD_PROJECTS WHERE PROJECT_ID = projectId;
	
	IF v_cnt = 0 THEN  
      INSERT INTO MD_PROJECTS
	  (PROJECT_ID, PROJECT_NAME, COMMENTS)
	  VALUES
	  (projectId, projectName, comments);
	  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' row inserted in MD_PROJECTS.');
  
	  INSERT INTO MD_CONNECTIONS 
	  (CONNECTION_ID, PROJECT_ID_FK, USERNAME, DBURL, NAME, TYPE)
	  VALUES
	  (connId, projectId, 'CONN_NAME', 'DBURL', 'DBNAME', 'CAPTURED');
	  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' row inserted in MD_CONNECTIONS.');
   END IF;	  
  
  COMMIT;
  END CaptureConnections;

  PROCEDURE CaptureDatabases(projectId IN NUMBER, connId IN NUMBER, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_ver NUMBER;
  BEGIN
      -- DBMS_OUTPUT.PUT_LINE('CaptureDatabases Procedure called');
   
   SELECT MAX(CAPT_VERSION) INTO v_ver 
   FROM  MD_CATALOGS 
   WHERE PROJECT_ID = projectId AND CONNECTION_ID = connId AND TYPE = 'CAPTURED';
	
   IF v_ver IS NULL THEN
	  v_ver := 1;
   ELSE
	  v_ver := v_ver + 1; -- increase the version number by 1.
   END IF;
   v_captver := v_ver;
  
  v_sql := q'[
  INSERT
  INTO
    MD_CATALOGS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CAPT_VERSION,
      CONV_VERSION,
      CATALOG_NAME,
      DUMMY_FLAG,
      TYPE
    )
    (
      -- TODO WE NEED STAGE TABLE FOR SERVER
      -- WIP  
      SELECT DISTINCT
        :projectId,
        :connId,
        :v_captver,
        :v_conver,
        CATALOG_NAME,
        'N',
        'CAPTURED'
      FROM
        MD_STAGE_PG8_DATABASES         
    )]';
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);      
  EXECUTE IMMEDIATE v_sql USING projectId, connId, v_captver, v_conver;      
  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' row inserted in MD_CATALOGS.');
  COMMIT;
  END CaptureDatabases;
  
  PROCEDURE CaptureSchemas(projectId IN NUMBER, connId IN NUMBER, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  
  v_sql CLOB;
  v_schemas CLOB;
  cur NUMBER;
  ex NUMBER;
  BEGIN
  v_schemas := convertArrToStr(schemaArr);
    
  v_sql := q'[   
  INSERT
  INTO
    MD_SCHEMAS
    (
      CONNECTION_ID,
      PROJECT_ID,
      CATALOG_NAME,
      SCHEMA_NAME,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      USER_NAMESPACE
    )
    (
      SELECT
        :svrId,
        :projectId,
        CATALOG_NAME,
        SCHEMA_NAME,
        'CAPTURED',
        :verId,
        :converId,
        'NS_DATABASE-' || SCHEMA_NAME || '-' || :svrId || '-' || :projectId
      FROM
        MD_STAGE_PG8_DATABASES
    ]';
  
  cur := DBMS_SQL.OPEN_CURSOR;
  IF DBMS_LOB.GETLENGTH(v_schemas) = 0 THEN
    v_sql := v_sql || ')';
  ELSE
    v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
    v_sql := v_sql || v_schemas || ')';
  END IF;
  
  -- DBMS_OUTPUT.PUT_LINE(v_sql);
  DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
  DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_captver);
  DBMS_SQL.BIND_VARIABLE(cur, 'converId', v_conver);
  DBMS_SQL.BIND_VARIABLE(cur, 'svrId', connId);
  DBMS_SQL.BIND_VARIABLE(cur, 'projectId', projectId);
  ex := DBMS_SQL.EXECUTE(cur);
  DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_SCHEMAS.');
  DBMS_SQL.CLOSE_CURSOR(cur);
  COMMIT;
  END CaptureSchemas;

  -----------------------------------------------
  -- Capture Tables
  -----------------------------------------------
  PROCEDURE CaptureTables(projectId IN NUMBER, connId IN NUMBER, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;
  
  BEGIN
	  
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);
  
  v_sql := q'[	       
    INSERT INTO MD_TABLES
     (
        CONNECTION_ID,
        PROJECT_ID,
        TYPE,
        CAPT_VERSION,
        CONV_VERSION,
        CATALOG_NAME,
        SCHEMA_NAME,
        TABLE_NAME,
        QUALIFIED_NATIVE_NAME,
        SCHEMA_NAMESPACE,
        COMMENTS,
        NUM_ROWS
     )
     (
      SELECT 
        T1.CONNECTION_ID,
        T1.PROJECT_ID,
        T1.TYPE,
        T1.CAPT_VERSION,
        T1.CONV_VERSION,
        T1.CATALOG_NAME,
        T1.SCHEMA_NAME,
        T1.TABLE_NAME,
        T1.QUALIFIED_NATIVE_NAME,
        T1.SCHEMA_NAMESPACE,
        T1.COMMENTS,
        T1.NUM_ROWS
      FROM  
	   ( SELECT
	         :svrId CONNECTION_ID,
             :projectId PROJECT_ID,
	         'CAPTURED' TYPE,
             :verId CAPT_VERSION,
             :converId CONV_VERSION,
	         REGEXP_REPLACE(SPG8_TAB.CATALOG_NAME, CHR(10)||'(.*)', '\1') CATALOG_NAME,
	         SPG8_TAB.SCHEMA_NAME SCHEMA_NAME,
	         SPG8_TAB.TABLE_NAME TABLE_NAME,
	         SPG8_TAB.CATALOG_NAME || '.' || SPG8_TAB.SCHEMA_NAME || '.' || SPG8_TAB.TABLE_NAME QUALIFIED_NATIVE_NAME,
	         'NS_SCHEMAOBJ-' || SPG8_TAB.SCHEMA_NAME || '-' || :svrId || '-' || :projectId SCHEMA_NAMESPACE,
	         SPG8_COM.COMMENTS_TEXT COMMENTS,
	         SPG8_TAB.NUM_ROWS_IN_TABLE NUM_ROWS
	     FROM
	         MD_STAGE_PG8_TABLES  SPG8_TAB 
	     LEFT JOIN
	       MD_STAGE_PG8_COMMENTS SPG8_COM
	     ON
             SPG8_TAB.SCHEMA_NAME = SPG8_COM.SCHEMA_NAME AND
             SPG8_TAB.TABLE_NAME = SPG8_COM.OBJECT_NAME AND
             SPG8_COM.OBJECT_TYPE = 'table' AND
             SPG8_COM.PARENT_NAME IS NULL
         ) T1
    ]'; 
    
    cur := DBMS_SQL.OPEN_CURSOR;
	
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND LENGTH(v_tables) > 0 THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;
    
	v_sql := v_sql || ')';
	-- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_captver);
    DBMS_SQL.BIND_VARIABLE(cur, 'converId', v_conver);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', connId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', projectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_TABLES.');
    DBMS_SQL.CLOSE_CURSOR(cur);
    COMMIT;
  END CaptureTables;


  -----------------------------------------------
  -- Capture Columns
  -----------------------------------------------
  PROCEDURE CaptureColumns(projectId IN NUMBER, connId IN NUMBER, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;
  BEGIN
  -- DBMS_OUTPUT.PUT_LINE('CaptureColumns Procedure called');
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);
  
  v_sql := q'[
    INSERT INTO MD_COLUMNS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      SCHEMA_NAME,
      TABLE_NAME,
      COLUMN_NAME,
      COLUMN_ORDER,
      COLUMN_TYPE,
      PRECISION,
      SCALE,
      DEFAULT_VALUE,
      TYPE,
      CAPT_VERSION,
      CONV_VERSION,
      COLUMN_NAMESPACE,
      COMMENTS
    )
    (
      SELECT
        PROJECT_ID,
        CONNECTION_ID,
        CATALOG_NAME,
        SCHEMA_NAME,
        TABLE_NAME,
        COLUMN_NAME,
        COLUMN_ORDER,
        COLUMN_TYPE,
        PRECISION,
        SCALE,
        DEFAULT_VALUE,
        'CAPTURED',
        :verId CAPT_VERSION,
        :converId CONV_VERSION,
        COLUMN_NAMESPACE,
        COMMENTS
      FROM
       (
          SELECT
             T1.PROJECT_ID PROJECT_ID,   T1.CONNECTION_ID CONNECTION_ID,
             T1.CATALOG_NAME CATALOG_NAME,  T1.SCHEMA_NAME SCHEMA_NAME, 
             T1.TABLE_NAME TABLE_NAME, T1.COLUMN_NAME COLUMN_NAME, 
             T1.COLUMN_ORDER COLUMN_ORDER, T1.COLUMN_TYPE COLUMN_TYPE,
             T1.PRECISION PRECISION, T1.SCALE SCALE, T1.DEFAULT_VALUE DEFAULT_VALUE, 
             T1.COLUMN_NAMESPACE COLUMN_NAMESPACE, T1.COMMENTS COMMENTS
          FROM 
          (
             SELECT
               :projectId PROJECT_ID,
               :svrId CONNECTION_ID,
               REGEXP_REPLACE(SPG8_COLS.CATALOG_NAME, CHR(10)||'(.*)', '\1') CATALOG_NAME,
               REGEXP_REPLACE(SPG8_COLS.SCHEMA_NAME, CHR(10)||'(.*)', '\1') SCHEMA_NAME,
               REGEXP_REPLACE(SPG8_COLS.TABLE_NAME, CHR(10)||'(.*)', '\1') TABLE_NAME,
               REGEXP_REPLACE(COLUMN_NAME, CHR(10)||'(.*)', '\1') COLUMN_NAME,
               ORDINAL_POSITION COLUMN_ORDER,
               DATA_TYPE COLUMN_TYPE,
               -- Interval types needs to be resolved.
               CASE
                WHEN DATA_TYPE = 'timestamp without time zone' OR
                  DATA_TYPE = 'timestamp with time zone' OR
                  DATA_TYPE = 'date' OR
                  DATA_TYPE = 'time without time zone' OR
                  DATA_TYPE = 'time with time zone'
                THEN DATETIME_PRECISION
                WHEN DATA_TYPE = 'character varying' OR
                  DATA_TYPE = 'character' OR
                  DATA_TYPE = 'char' OR
                  DATA_TYPE = 'varchar'
                THEN CHARACTER_MAXIMUM_LENGTH
                ELSE NUMERIC_PRECISION 
               END PRECISION,
               NUMERIC_SCALE SCALE,
               /*CASE
                WHEN IS_NULLABLE = 'YES'
                THEN 'Y'
                ELSE 'N'
                END NULLABLE,*/
               COLUMN_DEFAULT DEFAULT_VALUE,
               'NS_COLUMN-'
                || SPG8_COLS.SCHEMA_NAME || '-'
                || SPG8_COLS.TABLE_NAME || '-' || :svrId || '-' || :projectId COLUMN_NAMESPACE,
               SPG8_COM.COMMENTS_TEXT COMMENTS
             FROM
               MD_STAGE_PG8_COLUMNS SPG8_COLS 
             LEFT JOIN
               MD_STAGE_PG8_COMMENTS SPG8_COM 
             ON
               SPG8_COLS.SCHEMA_NAME = SPG8_COM.SCHEMA_NAME AND
               SPG8_COLS.TABLE_NAME = SPG8_COM.PARENT_NAME AND
               SPG8_COLS.COLUMN_NAME = SPG8_COM.OBJECT_NAME
             ORDER BY
               TABLE_NAME, COLUMN_NAME
          ) T1      
       )             
    ]';
      
    cur := DBMS_SQL.OPEN_CURSOR;
	
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;
	
	v_sql := v_sql || ')';
    
	-- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'verId', v_captver);
    DBMS_SQL.BIND_VARIABLE(cur, 'converId', v_conver);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', connId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', projectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_COLUMNS.');
    DBMS_SQL.CLOSE_CURSOR(cur);
    COMMIT;
  END CaptureColumns;

  ---------------------------------------------------
  -- Capture Constraints
  ---------------------------------------------------
  PROCEDURE CaptureConstraints(projectId IN NUMBER, connId IN NUMBER, 
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB := EMPTY_CLOB();
  v_schemas CLOB := EMPTY_CLOB();
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;
  
  BEGIN
    -- DBMS_OUTPUT.PUT_LINE('CaptureConstraints Procedure called');
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);
  
  v_sql := q'[
	INSERT INTO MD_CONSTRAINTS
    (
       PROJECT_ID,
       CONNECTION_ID,
       CATALOG_NAME,
       TYPE,
       SCHEMA_NAME,
       TABLE_NAME,
       CONSTRAINT_NAME,
       COLUMN_NAME,
       DETAIL_ORDER,
       REFCATALOG_NAME,
       REFSCHEMA_NAME,
       REFTABLE_NAME,
       REFCOLUMN_NAME,
       CONSTRAINT_TEXT,
       LANGUAGE,
       CONSTRAINT_TYPE,
       CONSTRAINT_NAMESPACE,
       COMMENTS
    )
    (
       SELECT
          T1.PROJECT_ID,
          T1.CONNECTION_ID,
          T1.CATALOG_NAME,
          T1.TYPE,
          T1.SCHEMA_NAME,
          T1.TABLE_NAME,
          T1.CONSTRAINT_NAME,
          T1.COLUMN_NAME,
          T1.DETAIL_ORDER,
          T1.REFCATALOG_NAME,
          T1.REFSCHEMA_NAME,
          T1.REFTABLE_NAME,
          T1.REFCOLUMN_NAME,
          T1.CONSTRAINT_TEXT,
          T1.LANGUAGE,
          T1.CONSTRAINT_TYPE,
          T1.CONSTRAINT_NAMESPACE,
          T1.COMMENTS
       FROM (
          SELECT
             :projectId PROJECT_ID,
             :svrId CONNECTION_ID,
             REGEXP_REPLACE(SPG8_CONS.CATALOG_NAME, CHR(10)||'(.*)', '\1') CATALOG_NAME,
             'CAPTURED' TYPE,
             SPG8_CONS.SCHEMA_NAME,
             TABLE_NAME,
             CONSTRAINT_NAME,
             COLUMN_NAME,
             ORDINAL_POSITION DETAIL_ORDER,
             REFCATALOG_NAME REFCATALOG_NAME,
             REFSCHEMA_NAME REFSCHEMA_NAME,
             REFTABLE_NAME REFTABLE_NAME,
             REFCOLUMN_NAME REFCOLUMN_NAME,
             CONSTRAINT_CLAUSE CONSTRAINT_TEXT,
             'STSQL' LANGUAGE,
             CASE 
                WHEN CONSTRAINT_TYPE = 'PRIMARY KEY' THEN 'PK'
                ELSE CONSTRAINT_TYPE
             END CONSTRAINT_TYPE,
             'NS_CONSTRAINTS-' || SPG8_CONS.SCHEMA_NAME || '-' || :svrId || '-' || :projectId CONSTRAINT_NAMESPACE,
             SPG8_COM.COMMENTS_TEXT COMMENTS
          FROM
             MD_STAGE_PG8_CONSTRAINTS SPG8_CONS
          LEFT JOIN MD_STAGE_PG8_COMMENTS SPG8_COM ON
             SPG8_CONS.CATALOG_NAME = SPG8_COM.CATALOG_NAME AND
             SPG8_CONS.SCHEMA_NAME = SPG8_COM.SCHEMA_NAME AND
             SPG8_CONS.CONSTRAINT_NAME = SPG8_COM.OBJECT_NAME
             AND SPG8_COM.OBJECT_TYPE in ('primary_key', 'foreign_key', 'composite_type', 'check' )
          ) T1
   ]';
   
   cur := DBMS_SQL.OPEN_CURSOR;
	
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;
	
	v_sql := v_sql || ')';
    
	DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', connId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', projectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_CONSTRAINTS.');
    DBMS_SQL.CLOSE_CURSOR(cur);
   
	-- );
	-- DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_CONSTRAINTS.');
    COMMIT;
  END CaptureConstraints;

  ---------------------------------------------------
  -- Capture Indexes
  ---------------------------------------------------
  PROCEDURE CaptureIndexes(projectId IN NUMBER, connId IN NUMBER,  
           schemaArr IN NAMES DEFAULT NAMES(NULL), tableArr IN NAMES DEFAULT NAMES(NULL)) AS
  v_sql CLOB;
  v_schemas CLOB;
  v_tables CLOB;
  cur NUMBER;
  ex NUMBER;         
  BEGIN
   -- DBMS_OUTPUT.PUT_LINE('CaptureIndexes Procedure called');
  v_schemas := convertArrToStr(schemaArr);
  v_tables := convertArrToStr(tableArr);
  
  v_sql := q'[
    INSERT INTO MD_INDEXES
    (
       PROJECT_ID,
       CONNECTION_ID, 
       CATALOG_NAME, 
       SCHEMA_NAME, 
       TABLE_NAME, 
       COLUMN_NAME, 
       INDEX_NAME,
       DETAIL_ORDER,
       INDEX_NAMESPACE, 
       TYPE,
       COMMENTS
    )
    (
       SELECT 
          T1.PROJECT_ID,
          T1.CONNECTION_ID, 
          T1.CATALOG_NAME, 
          T1.SCHEMA_NAME, 
          T1.TABLE_NAME, 
          T1.COLUMN_NAME, 
          T1.INDEX_NAME,
          T1.DETAIL_ORDER,
          T1.INDEX_NAMESPACE, 
          T1.TYPE,
          T1.COMMENTS
       FROM 
       (
          SELECT
             :projectId PROJECT_ID, 
             :svrId CONNECTION_ID, 
             REGEXP_REPLACE(SPG8_IDX.CATALOG_NAME, CHR(10)||'(.*)', '\1') CATALOG_NAME,
             SPG8_IDX.SCHEMA_NAME SCHEMA_NAME, 
             TABLE_NAME, 
             COLUMN_NAME, 
             INDEX_NAME,
             -1 DETAIL_ORDER,
             'NS_INDEXES-' || SPG8_IDX.SCHEMA_NAME || '-' || TO_CHAR(:svrId) || '-' || TO_CHAR(:projectId) INDEX_NAMESPACE, 
             'CAPTURED' TYPE,
             SPG8_COM.COMMENTS_TEXT COMMENTS
          FROM
             MD_STAGE_PG8_INDEXES SPG8_IDX, MD_STAGE_PG8_COMMENTS SPG8_COM
          WHERE
             SPG8_IDX.SCHEMA_NAME = SPG8_COM.SCHEMA_NAME AND
             SPG8_IDX.INDEX_NAME = SPG8_COM.OBJECT_NAME AND
             SPG8_IDX.TABLE_NAME = SPG8_COM.PARENT_NAME
       ) T1
   ]';
    
    cur := DBMS_SQL.OPEN_CURSOR;
	IF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) = 0  THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_sql := v_sql || v_schemas;
	ELSIF DBMS_LOB.GETLENGTH(v_schemas) > 0 AND DBMS_LOB.GETLENGTH(v_tables) > 0 THEN
	  v_schemas := ' WHERE UPPER(SCHEMA_NAME) IN(' || v_schemas || ')';
	  v_tables := ' AND UPPER(TABLE_NAME) IN(' || v_tables || ')';
	  v_sql := v_sql || v_schemas || v_tables;
	ELSE
	  v_sql := v_sql;
	END IF;
	
	v_sql := v_sql || ')';
    
	-- DBMS_OUTPUT.PUT_LINE(v_sql);
    DBMS_SQL.PARSE(cur, v_sql, DBMS_SQL.NATIVE);
    DBMS_SQL.BIND_VARIABLE(cur, 'svrId', connId);
    DBMS_SQL.BIND_VARIABLE(cur, 'projectId', projectId);
    ex := DBMS_SQL.EXECUTE(cur);
    DBMS_OUTPUT.PUT_LINE(DBMS_SQL.LAST_ROW_COUNT || ' rows inserted in MD_INDEXES.');
    DBMS_SQL.CLOSE_CURSOR(cur);
    COMMIT;
  END CaptureIndexes;
  
  
  /* No comments are added for sequences, functions, views, and triggers as these procedure are commented out for Redshift. */
  /****************************************************************************
  PROCEDURE CaptureSequences(connId IN NUMBER, projectId IN NUMBER) AS
  BEGIN
    INSERT INTO MD_SEQUENCES
  (
     PROJECT_ID,
     CONNECTION_ID, 
     CATALOG_NAME,
     TYPE,
     SCHEMA_NAME, 
     SEQUENCE_NAME,
     SEQ_START,
     CURRENT_VALUE,
     INCR,
     MIN,
     "ORDER",
     SCHEMA_NAMESPACE
  )
    (
       SELECT
          projectId PROJECT_ID, 
          connId CONNECTION_ID, 
          REGEXP_REPLACE(SEQUENCE_CATALOG, CHR(10)||'(.*)', '\1') CATALOG_NAME,
          'CAPTURED' TYPE,
          SEQUENCE_SCHEMA SCHEMA_NAME, 
          SEQUENCE_NAME,
          START_VALUE SEQ_START,
          START_VALUE CURRENT_VALUE, -- using it for current_value for now.
          INCRMENT INCR,
          MINIMUM_VALUE MIN,
          2,
          'NS_SCHEMAOBJ-' || SEQUENCE_SCHEMA || '-' || connId || '-' || projectId
       FROM
          S_PG8_SEQUENCES
    );
    
    DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_SEQUENCES.');
    COMMIT;
  END CaptureSequences;

  PROCEDURE CaptureViews(connId IN NUMBER, projectId IN NUMBER) AS
  BEGIN
      -- DBMS_OUTPUT.PUT_LINE('CaptureViews Procedure called');
  INSERT
  INTO
    MD_VIEWS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      SCHEMA_NAME,
      VIEW_NAME,
      LANGUAGE,
      NATIVE_SQL,
      LINECOUNT,
      TYPE,
      SCHEMA_NAMESPACE
    )
    (
       SELECT 
         projectId PROJECT_ID, 
         connId CONNECTION_ID,
         REGEXP_REPLACE(TABLE_CATALOG, CHR(10)||'(.*)', '\1') CATALOG_NAME,
         TABLE_SCHEMA SCHEMA_NAME,
         TABLE_NAME VIEW_NAME,
         'STSQL' LANGUAGE,
         VIEW_DEFINITION NATIVE_SQL,
         (LENGTH(LTRIM(RTRIM(VIEW_DEFINITION,' '
	      ||'  '
	      || CHR(10)
	      ||CHR(13)),' '
	      ||'  '
	      || CHR(10)
	      ||CHR(13))) - LENGTH(REPLACE(LTRIM(RTRIM(VIEW_DEFINITION,' '
	      ||'  '
	      || CHR(10)
	      ||CHR(13)),' '
	      ||'  '
	      || CHR(10)
	      ||CHR(13)), CHR(10))) + 1) LINECOUNT,
         'CAPTURED' TYPE,
         'NS_SCHEMAOBJ-' || TABLE_SCHEMA || '-' || TO_CHAR(connId) || '-' || TO_CHAR(projectId) SCHEMA_NAMESPACE
        FROM
          S_PG8_VIEWS
    );
  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_VIEWS.');
  COMMIT;
  END CaptureViews;

  PROCEDURE CaptureStoredPrograms(connId IN NUMBER, projectId IN NUMBER) AS
  BEGIN
    -- DBMS_OUTPUT.PUT_LINE('CaptureStoredPrograms Procedure called');
  INSERT
  INTO
    MD_STORED_PROGRAMS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      SCHEMA_NAME,
      PROGRAM_NAME,
      NATIVE_SQL,
      LINECOUNT,
      LANGUAGE,
      TYPE,
      SCHEMA_NAMESPACE
    )
      (  
        SELECT 
		  PROC.PROJECT_ID,
		  PROC.CONNECTION_ID,
		  PROC.CATALOG_NAME,
		  PROC.SCHEMA_NAME,
		  DECODE( (ROW_NUMBER() OVER (PARTITION BY UPPER(PROC.PROGRAM_NAME), PROC.SCHEMA_NAMESPACE ORDER BY PROC.PROGRAM_NAME)), 
               1, PROC.PROGRAM_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
               MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(PROC.PROGRAM_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(PROC.PROGRAM_NAME), PROC.SCHEMA_NAMESPACE ORDER BY PROC.PROGRAM_NAME)), 128) ) PROGRAM_NAME,
	      PROC.NATIVE_SQL,
		  PROC.LINECOUNT,
		  PROC.LANGUAGE,
		  PROC.TYPE,
		  PROC.SCHEMA_NAMESPACE
	    FROM 
          (
		    SELECT
		      projectId PROJECT_ID,
		      connId CONNECTION_ID,
		      REGEXP_REPLACE(CATALOGNAME, CHR(10)||'(.*)', '\1') CATALOG_NAME,
		      SCHEMANAME SCHEMA_NAME,
		      PROCNAME PROGRAM_NAME,
		      FUNCTION_DEFINITION NATIVE_SQL,
		      (LENGTH(LTRIM(RTRIM(FUNCTION_DEFINITION,' '
		      ||'  '
		      || CHR(10)
		      ||CHR(13)),' '
		      ||'  '
		      || CHR(10)
		      ||CHR(13))) - LENGTH(REPLACE(LTRIM(RTRIM(FUNCTION_DEFINITION,' '
		      ||'  '
		      || CHR(10)
		      ||CHR(13)),' '
		      ||'  '
		      || CHR(10)
		      ||CHR(13)), CHR(10))) + 1) LINECOUNT,
		      'STSQL' LANGUAGE,
		      'CAPTURED' TYPE,
		      'NS_SCHEMAOBJ-' || SCHEMANAME || '-' || TO_CHAR(connId) || '-' || TO_CHAR(projectId) SCHEMA_NAMESPACE
		    FROM
		      S_PG8_PROCEDURES
		  ) PROC
        
   );
    
  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_STORED_PROGRAMS.');
  COMMIT;
  END CaptureStoredPrograms;

  PROCEDURE CaptureTriggers(connId IN NUMBER, projectId IN NUMBER) AS
  BEGIN
      -- DBMS_OUTPUT.PUT_LINE('CaptureTriggers Procedure called');
  INSERT
  INTO
    MD_TRIGGERS
    (
      PROJECT_ID,
      CONNECTION_ID,
      CATALOG_NAME,
      SCHEMA_NAME,
      TRIGGER_NAME,
      TRIGGER_ON_FLAG,
      TABLE_OR_VIEW_NAME,
      NATIVE_SQL,
      LINECOUNT,
      LANGUAGE,
      TRIGGER_OPERATION,
      TYPE,
      TRIGGER_NAMESPACE
    )
    (
       SELECT
         TRIG.PROJECT_ID,
	     TRIG.CONNECTION_ID,
	     TRIG.CATALOG_NAME,
	     TRIG.SCHEMA_NAME,
	     DECODE( (ROW_NUMBER() OVER (PARTITION BY UPPER(TRIG.TRIGGER_NAME), TRIG.TRIGGER_NAMESPACE ORDER BY TRIG.TRIGGER_NAME)), 
               1, TRIG.TRIGGER_NAME, -- if NOT unique then modify the Identifier by uppercasing it and appending with '_' and rownumber for now.
               MIGRATION_TRANSFORMER.ADD_SUFFIX(UPPER(TRIG.TRIGGER_NAME), '_' || TO_CHAR(ROW_NUMBER() OVER (PARTITION BY UPPER(TRIG.TRIGGER_NAME), TRIG.TRIGGER_NAMESPACE ORDER BY TRIG.TRIGGER_NAME)), 128) ) TRIGGER_NAME,
	     TRIG.TRIGGER_ON_FLAG,
	     TRIG.TABLE_OR_VIEW_NAME,
	     TRIG.NATIVE_SQL,
	     TRIG.LINECOUNT,
	     TRIG.LANGUAGE,
	     TRIG.TRIGGER_OPERATION,
	     TRIG.TYPE,
	     TRIG.TRIGGER_NAMESPACE
	   FROM
	    (
	      SELECT
	        projectId PROJECT_ID,
	        connId CONNECTION_ID,
	        REGEXP_REPLACE(TRIGGER_CATALOG, CHR(10)||'(.*)', '\1') CATALOG_NAME,
	        TRIGGER_SCHEMA SCHEMA_NAME,
	        TRIGGER_NAME TRIGGER_NAME,
	        'T' TRIGGER_ON_FLAG,
	        EVENT_OBJECT_TABLE TABLE_OR_VIEW_NAME,
	        ACTION_STATEMENT NATIVE_SQL,
	        (LENGTH(LTRIM(RTRIM(ACTION_STATEMENT,' '
	        ||'  '
	        || CHR(10)
	        ||CHR(13)),' '
	        ||'  '
	        || CHR(10)
	        ||CHR(13))) - LENGTH(REPLACE(LTRIM(RTRIM(ACTION_STATEMENT,' '
	        ||'  '
	        || CHR(10)
	        ||CHR(13)),' '
	        ||'  '
	        || CHR(10)
	        ||CHR(13)), CHR(10))) + 1) LINECOUNT,
	        'STSQL' LANGUAGE,
	        EVENT_MANIPULATION TRIGGER_OPERATION,
	        'CAPTURED' TYPE,
	        'NS_DB_TRIGGERS-' || TRIGGER_SCHEMA || '-' || connId || '-' || projectId TRIGGER_NAMESPACE
	      FROM
	        S_PG8_TRIGGERS
	    ) TRIG
	)   
    ;
  
  DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_TRIGGERS.');
  COMMIT;
  END CaptureTriggers;
  ************************************************************/

  PROCEDURE DefaultTypeMaps(projectId IN NUMBER, connId IN NUMBER) AS
  BEGIN
    INSERT
  INTO
    MD_DATATYPE_TRANSFORM_MAP
    (
      PROJECT_ID,
      CONNECTION_ID,
      MAP_NAME
    )
    VALUES
    (
      projectId,
      connId,
      'DEFAULTMAP'
    );
  COMMIT;  
  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'BIGINT',
      NULL,
      NULL,
      'NUMBER',
      24,
      0
    );
  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INT8',
      NULL,
      NULL,
      'NUMBER',
      24,
      NULL
    );
   
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'BIT',
      NULL,
      NULL,
      'RAW',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'BIT VARYING',
      NULL,
      NULL,
      'RAW',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'VARBIT',
      NULL,
      NULL,
      'RAW',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'BOOLEAN',
      NULL,
      NULL,
      'CHAR',
      1,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'BOOL',
      NULL,
      NULL,
      'CHAR',
      1,
      NULL
    );
    
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'BYTEA',
      NULL,
      NULL,
      'BLOB',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'CHARACTER VARYING',
      NULL,
      NULL,
      'VARCHAR2',
      4000,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'VARCHAR',
      NULL,
      NULL,
      'VARCHAR2',
      4000,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'CHARACTER',
      NULL,
      NULL,
      'CHARACTER',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'CHAR',
      NULL,
      NULL,
      'CHAR',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TSVECTOR',
      NULL,
      NULL,
      'CLOB',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TSQUERY',
      NULL,
      NULL,
      'CLOB',
      NULL,
      NULL
    );
    
   
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'DATE',
      NULL,
      NULL,
      'DATE',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'DOUBLE PRECISION',
      NULL,
      NULL,
      'BINARY_DOUBLE',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'FLOAT8',
      NULL,
      NULL,
      'BINARY_DOUBLE',
      NULL,
      NULL
    );
  
   
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTEGER',
      NULL,
      NULL,
      'NUMBER',
      12,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INT',
      NULL,
      NULL,
      'NUMBER',
      12,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INT4',
      NULL,
      NULL,
      'NUMBER',
      12,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL YEAR TO MONTH',
      NULL,
      NULL,
      'INTERVAL YEAR TO MONTH',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL DAY TO HOUR',
      NULL,
      NULL,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL DAY TO MINUTE',
      NULL,
      NULL,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL HOUR TO MINUTE',
      NULL,
      NULL,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL HOUR TO SECOND',
      NULL,
      NULL,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INTERVAL MINUTE TO SECOND',
      NULL,
      NULL,
      'INTERVAL DAY TO SECOND',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'JSON',
      NULL,
      NULL,
      'CLOB',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'JSONB',
      NULL,
      NULL,
      'BLOB',
      NULL,
      NULL
    );
   
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'MONEY',
      NULL,
      NULL,
      'NUMBER',
      17,
      2
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'NUMERIC',
      NULL,
      NULL,
      'NUMBER',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'DECIMAL',
      NULL,
      NULL,
      'NUMBER',
      NULL,
      NULL
    );
  
    
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'REAL',
      NULL,
      NULL,
      'FLOAT',
      24,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'FLOAT4',
      NULL,
      NULL,
      'FLOAT',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'SMALLINT',
      NULL,
      NULL,
      'NUMBER',
      7,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INT2',
      NULL,
      NULL,
      'NUMBER',
      7,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TEXT',
      NULL,
      NULL,
      'CLOB',
      NULL,
      NULL
    );
   
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIME',
      NULL,
      NULL,
      'TIMESTAMP',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIME WITHOUT TIME ZONE',
      NULL,
      NULL,
      'TIMESTAMP',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIME WITH TIME ZONE',
      NULL,
      NULL,
      'TIMESTAMP WITH TIME ZONE',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIMETZ',
      NULL,
      NULL,
      'TIMESTAMP WITH TIME ZONE',
      NULL,
      NULL
    );  
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIMESTAMP WITHOUT TIME ZONE',
      NULL,
      NULL,
      'TIMESTAMP',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIMESTAMP WITH TIME ZONE',
      NULL,
      NULL,
      'TIMESTAMP WITH TIME ZONE',
      NULL,
      NULL
    );
    
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'TIMESTAMPTZ',
      NULL,
      NULL,
      'TIMESTAMP WITH TIME ZONE',
      NULL,
      NULL
    );
 
  
  /* -- TEMPLATE 
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      '',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */
    
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'box',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */
  
  /*
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'cidr',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */ 
  
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'circle',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */  
   
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'INET',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */
    
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'line',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */ 
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'line',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
    */
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'lseg',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */ 
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'macaddr',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */  
    
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'path',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */ 
  
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'pg_lsn',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */ 
  
  /*
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'point',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
   */ 
  
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'polygon',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */  
  
  /*  
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'tsquery',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */  
  /*
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'tsvector',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */  
  
  /*
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'txid_snapshot',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */  
  
  /*
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'uuid',
      NULL,
      NULL,
      '',
      NULL,
      NULL
    );
  */  
  
  /*
  INSERT
  INTO
    MD_DATATYPE_TRANSFORM_RULE
    (
      MAP_NAME,
      PROJECT_ID,
      CONNECTION_ID,
      SOURCE_DATA_TYPE_NAME,
      SOURCE_PRECISION,
      SOURCE_SCALE,
      TARGET_DATA_TYPE_NAME,
      TARGET_PRECISION,
      TARGET_SCALE
    )
    VALUES
    (
      'DEFAULTMAP',
      projectId,
      connId,
      'XML',
      NULL,
      NULL,
      'XMLTYPE',
      NULL,
      NULL
    );
 */
  
  -- DBMS_OUTPUT.PUT_LINE(SQL%ROWCOUNT || ' rows inserted in MD_DATATYPE_TRANSFORM_RULE.');  
  COMMIT;
  END DefaultTypeMaps;

  FUNCTION CAPTURE(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2, 
           schemaArr IN REDSHIFT_CAPTURE.NAMES, tableArr IN REDSHIFT_CAPTURE.NAMES DEFAULT NAMES(NULL)) RETURN VARCHAR2 AS
   v_startTime   TIMESTAMP;
   v_endTime     TIMESTAMP;

  BEGIN
     v_startTime := CURRENT_TIMESTAMP;
  -- v_start := DBMS_UTILITY.GET_TIME;
  CaptureConnections(projectId, connId, projectName , comments );
  CaptureDatabases(projectId, connId);
  CaptureSchemas(projectId, connId, schemaArr);
  CaptureTables(projectId, connId, schemaArr, tableArr);
  CaptureColumns(projectId, connId, schemaArr, tableArr);
  -- We don't need to Capture keys(PK, FK and UK) for ARS.
  -- CaptureConstraints(connId,projectId, schemaArr, tableArr);
  -- The following procedures will be enabled if required.
  -- CaptureIndexes(connId,projectId, schemaArr, tableArr);
  -- CaptureViews(connId,projectId);
  -- CaptureStoredPrograms(connId,projectId);
  -- CaptureTriggers(connId,projectId);
  -- CaptureSequences(connId,projectId);
  -- CaptureUDT;
  DefaultTypeMaps(projectId, connId);
  v_endTime := CURRENT_TIMESTAMP;
  return TO_CHAR(v_endTime - v_startTime);
  END CAPTURE;
  
  FUNCTION CAPTURE(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2, 
           schemaJArr IN VARCHAR2, tableJArr IN VARCHAR2 DEFAULT '[]') RETURN VARCHAR2 AS
  v_schemaArr REDSHIFT_CAPTURE.NAMES;
  v_tableArr REDSHIFT_CAPTURE.NAMES;
  BEGIN
	  v_schemaArr := convertJSONArrStrToNames(schemaJArr);
	  v_tableArr := convertJSONArrStrToNames(tableJArr);
	  RETURN CAPTURE(projectId, connId, projectName, comments, v_schemaArr, v_tableArr);
  END CAPTURE;	  


  FUNCTION CAPTURE(projectId IN NUMBER, connId IN NUMBER, projectName IN VARCHAR2, comments in VARCHAR2 ) RETURN VARCHAR2 AS
  BEGIN
	  RETURN CAPTURE(projectId, connId, projectName, comments, REDSHIFT_CAPTURE.NAMES(NULL), REDSHIFT_CAPTURE.NAMES(NULL));
  END CAPTURE;	  

END;
/
/*
Renamed object names 

STAGE_CONSTRAINTS -> MD_STAGE_CONSTRAINTS
STAGE_INDEXES     -> MD_STAGE_INDEXES
S_PG8_DATABASES   -> MD_STAGE_PG8_DATABASES
S_PG8_TABLES  	  -> MD_STAGE_PG8_TABLES
S_PG8_COLUMNS	  -> MD_STAGE_PG8_COLUMNS
S_PG8_CONSTRAINTS -> MD_STAGE_PG8_CONSTRAINTS
S_PG8_INDEXES	  -> MD_STAGE_PG8_INDEXES
S_PG8_COMMENTS	  -> MD_STAGE_PG8_COMMENTS

*/



/*
PROCEDURE create_MD_STAGE_PG8_DATABASES
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		
		
	]');
	
END create_MD_STAGE_PG8_DATABASES;

PROCEDURE create_MD_STAGE_PG8_DATABASES
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		
		
	]');
	
END create_MD_STAGE_PG8_DATABASES;

PROCEDURE create_MD_STAGE_PG8_DATABASES
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		
		
	]');
	
END create_MD_STAGE_PG8_DATABASES;
*/
