/***************************************************************************
 *
* Copyright (c) 2014, Oracle and/or its affiliates. All rights reserved.
 *
 * This is the PL/SQL block that implements the type mapping and
 * compression evaluation functions used by ttImportFromOracle
 * and SQL Developer.
 *
 * It is used to generate the C source file, 'plsql.c', used to build
 * ttImportFromOracle.
 *
 ***************************************************************************/

DECLARE

-- General constants
    resultSuccess CONSTANT NUMBER := 0;
    resultSkipTable CONSTANT NUMBER := 1;
    resultError CONSTANT NUMBER := 2;
    maxSQLNameLen CONSTANT PLS_INTEGER := 30;
    maxMsgSize CONSTANT PLS_INTEGER := 512;
    maxColdataSize CONSTANT PLS_INTEGER := 512;
    qt CONSTANT CHAR(1) := CHR(34); -- double quote
    sqt CONSTANT CHAR(1) := CHR(39); -- single quote

--  DEBUG related stuff
    debugTableName CONSTANT VARCHAR2(30) := 'DEBUG';
    debugSequenceName CONSTANT VARCHAR2(30) := 'DBGSEQ';
    debugMode CONSTANT BOOLEAN := false;

-- Oracle datatype constants
--     These *must* be maintained in sync with the values defined 
--     in 'ora2ttddl.h'
    oraTypeUnknown CONSTANT PLS_INTEGER := 0;
    oraTypeInvalid CONSTANT PLS_INTEGER := 1;
    oraTypeVarchar2 CONSTANT PLS_INTEGER := 2;
    oraTypeNvarchar2 CONSTANT PLS_INTEGER := 3;
    oraTypeNumber CONSTANT PLS_INTEGER := 4;
    oraTypeFloat CONSTANT PLS_INTEGER := 5;
    oraTypeDate CONSTANT PLS_INTEGER := 6;
    oraTypeBinaryFloat CONSTANT PLS_INTEGER := 7;
    oraTypeBinaryDouble CONSTANT PLS_INTEGER := 8;
    oraTypeTimestamp CONSTANT PLS_INTEGER := 9;
    oraTypeTimestampTZ CONSTANT PLS_INTEGER := 10;
    oraTypeTimestampLTZ CONSTANT PLS_INTEGER := 11;
    oraTypeChar CONSTANT PLS_INTEGER := 12;
    oraTypeNchar CONSTANT PLS_INTEGER := 13;
    oraTypeClob CONSTANT PLS_INTEGER := 14;
    oraTypeNclob CONSTANT PLS_INTEGER := 15;
    oraTypeBlob CONSTANT PLS_INTEGER := 16;
    oraTypeRaw CONSTANT PLS_INTEGER := 17;
    oraTypeLong CONSTANT PLS_INTEGER := 18;
    oraTypeLongraw CONSTANT PLS_INTEGER := 19;

    oraTypeUnknownS CONSTANT VARCHAR2(16) := '*UNKNOWN*';
    oraTypeInvalidS CONSTANT VARCHAR2(16) := '*INVALID*';
    oraTypeVarchar2S CONSTANT VARCHAR2(16) := 'VARCHAR2';
    oraTypeNvarchar2S CONSTANT VARCHAR2(16) := 'NVARCHAR2';
    oraTypeNumberS CONSTANT VARCHAR2(16) := 'NUMBER';
    oraTypeFloatS CONSTANT VARCHAR2(16) := 'FLOAT';
    oraTypeDateS CONSTANT VARCHAR2(16) := 'DATE';
    oraTypeBinaryFloatS CONSTANT VARCHAR2(16) := 'BINARY_FLOAT';
    oraTypeBinaryDoubleS CONSTANT VARCHAR2(16) := 'BINARY_DOUBLE';
    oraTypeTimestampS CONSTANT VARCHAR2(16) := 'TIMESTAMP';
    oraTypeCharS CONSTANT VARCHAR2(16) := 'CHAR';
    oraTypeNcharS CONSTANT VARCHAR2(16) := 'NCHAR';
    oraTypeClobS CONSTANT VARCHAR2(16) := 'CLOB';
    oraTypeNclobS CONSTANT VARCHAR2(16) := 'NCLOB';
    oraTypeBlobS CONSTANT VARCHAR2(16) := 'BLOB';
    oraTypeRawS CONSTANT VARCHAR2(16) := 'RAW';
    oraTypeLongS CONSTANT VARCHAR2(16) := 'LONG';
    oraTypeLongrawS CONSTANT VARCHAR2(16) := 'LONG RAW';

    oraTypeWithTZ CONSTANT VARCHAR2(32) := ' WITH TIME ZONE';
    oraTypeWithLTZ CONSTANT VARCHAR2(32) := ' WITH LOCAL TIME ZONE';

    oraRawMaxlen CONSTANT PLS_INTEGER := 2000;
    oraFloatMinPrecision CONSTANT PLS_INTEGER := 1;
    oraFloatMaxPrecision CONSTANT PLS_INTEGER := 126;
    oraNumberMinPrecision CONSTANT PLS_INTEGER := 1;
    oraNumberMaxPrecision CONSTANT PLS_INTEGER := 38;
    oraNumberMinScale CONSTANT PLS_INTEGER := -84;
    oraNumberMaxScale CONSTANT PLS_INTEGER := 127;
    oraTimestampMinScale CONSTANT PLS_INTEGER := 0;
    oraTimestampMaxScale CONSTANT PLS_INTEGER := 9;

-- TimesTen datatype constants
    ttTypeUnknown CONSTANT PLS_INTEGER := 0;
    ttTypeInvalid CONSTANT PLS_INTEGER := 1;
    ttTypeVarchar2 CONSTANT PLS_INTEGER := 2;
    ttTypeNvarchar2 CONSTANT PLS_INTEGER := 3;
    ttTypeNumber CONSTANT PLS_INTEGER := 4;
    ttTypeFloat CONSTANT PLS_INTEGER := 5;
    ttTypeDate CONSTANT PLS_INTEGER := 6;
    ttTypeBinaryFloat CONSTANT PLS_INTEGER := 7;
    ttTypeBinaryDouble CONSTANT PLS_INTEGER := 8;
    ttTypeTimestamp CONSTANT PLS_INTEGER := 9;
    ttTypeChar CONSTANT PLS_INTEGER := 10;
    ttTypeNchar CONSTANT PLS_INTEGER := 11;
    ttTypeClob CONSTANT PLS_INTEGER := 12;
    ttTypeNClob CONSTANT PLS_INTEGER := 13;
    ttTypeBlob CONSTANT PLS_INTEGER := 14;
    ttTypeTinyint CONSTANT PLS_INTEGER := 15;
    ttTypeSmallint CONSTANT PLS_INTEGER := 16;
    ttTypeInteger CONSTANT PLS_INTEGER := 17;
    ttTypeBigint CONSTANT PLS_INTEGER := 18;
    ttTypeVarbinary CONSTANT PLS_INTEGER := 19;
    ttTypeMin CONSTANT PLS_INTEGER := ttTypeVarchar2;
    ttTypeMax CONSTANT PLS_INTEGER := ttTypeVarbinary;

-- TimesTen type alignments
    ttAlignUnknown CONSTANT PLS_INTEGER := 0;
    ttAlignInvalid CONSTANT PLS_INTEGER := 0;
    ttAlignVarchar2 CONSTANT PLS_INTEGER := 8;
    ttAlignNvarchar2 CONSTANT PLS_INTEGER := 8;
    ttAlignNumber CONSTANT PLS_INTEGER := 1;
    ttAlignFloat CONSTANT PLS_INTEGER := 1;
    ttAlignDate CONSTANT PLS_INTEGER := 1;
    ttAlignBinaryFloat CONSTANT PLS_INTEGER := 4;
    ttAlignBinaryDouble CONSTANT PLS_INTEGER := 8;
    ttAlignTimestamp CONSTANT PLS_INTEGER := 4;
    ttAlignChar CONSTANT PLS_INTEGER := 1;
    ttAlignNchar CONSTANT PLS_INTEGER := 2;
    ttAlignClob CONSTANT PLS_INTEGER := 8;
    ttAlignNClob CONSTANT PLS_INTEGER := 8;
    ttAlignBlob CONSTANT PLS_INTEGER := 8;
    ttAlignTinyint CONSTANT PLS_INTEGER := 1;
    ttAlignSmallint CONSTANT PLS_INTEGER := 2;
    ttAlignInteger CONSTANT PLS_INTEGER := 4;
    ttAlignBigint CONSTANT PLS_INTEGER := 8;
    ttAlignVarbinary CONSTANT PLS_INTEGER := 8;

-- Other TimesTen constants
    ttMaxInlineLimit CONSTANT PLS_INTEGER := 32768;
    ttVarMaxlen CONSTANT PLS_INTEGER := 4194304;
    ttNvarMaxlen CONSTANT PLS_INTEGER := 2097152;
    ttFloatPrecisionCvt CONSTANT NUMBER := 3.316;
    ttPTPgBitsPerNull CONSTANT PLS_INTEGER := 1;
    ttPTPgSlotnoAlign CONSTANT PLS_INTEGER := 2;
    ttPTPgRefcntAlign CONSTANT PLS_INTEGER := 4;
    ttDfltVarlenUsage CONSTANT NUMBER := 0.5;
    ttPTPgTupAlign CONSTANT PLS_INTEGER := 8;
    ttEntriesPerLTPg CONSTANT PLS_INTEGER := 256;
    ttEntriesPerPTPg CONSTANT PLS_INTEGER := 256;
    ttLTPgRowSize CONSTANT PLS_INTEGER := 16;
    ttOverheadPerPage CONSTANT PLS_INTEGER := 520;
    ttMetaPerTable CONSTANT PLS_INTEGER := 500;
    ttMetaPerColumn CONSTANT PLS_INTEGER := 315;

-- Type definitions
    TYPE aligntable IS VARRAY(20) OF PLS_INTEGER;

    TYPE msgtable IS TABLE OF VARCHAR2(512) INDEX BY PLS_INTEGER;

    TYPE datatable IS TABLE OF VARCHAR2(512) INDEX BY PLS_INTEGER;

    TYPE qresult IS TABLE OF NUMBER INDEX BY PLS_INTEGER;

    TYPE columndef IS RECORD
    (
        name                VARCHAR2(30),
        skip                BOOLEAN := FALSE,
        oradatatype         PLS_INTEGER := NULL,
        oradatalen          PLS_INTEGER := NULL,
        oraprecision        PLS_INTEGER := NULL,
        orascale            PLS_INTEGER := NULL,
        oracharlength       PLS_INTEGER := NULL,
        oratypetext         VARCHAR2(64) := NULL,
        compressible        BOOLEAN := FALSE,
        alwaysoutofline     BOOLEAN := FALSE,
        nullable            BOOLEAN := FALSE,
        defaschar           BOOLEAN := NULL,
        isoutofline         BOOLEAN := NULL,
        wasoptimised        BOOLEAN := FALSE,
        hasfractionalvalues BOOLEAN := FALSE,
        hasnegativevalues   BOOLEAN := TRUE,
        ttdatatype          PLS_INTEGER := NULL,
        ttdatalen           PLS_INTEGER := NULL,
        ttprecision         PLS_INTEGER := NULL,
        ttscale             PLS_INTEGER := NULL,
        ttcharlength        PLS_INTEGER := NULL,
        ttptrsize           PLS_INTEGER := 0,
        ttalignment         PLS_INTEGER := 0,
        tttypetext          VARCHAR2(64) := NULL,
        avglength           NUMBER(38) := NULL,
        count               NUMBER(38) := NULL,
        countdistinct       NUMBER(38) := NULL,
        maxintegrallen      NUMBER(38) := NULL,
        maxfractionallen    NUMBER(38) := NULL,
        estcompressratio    NUMBER(38) := NULL
    );
    TYPE columnarray IS TABLE of columndef INDEX BY PLS_INTEGER;

    TYPE tupledef IS RECORD
    (
        numnullable             PLS_INTEGER := 0,
        inlinesize              NUMBER(38,0) := 0,
        outoflinesize           NUMBER(38,0) := 0,
        numcols                 PLS_INTEGER := 0,
        cols                    columnarray
    );

    TYPE tabledef IS RECORD
    (
        owner                   VARCHAR2(30),
        name                    VARCHAR2(30),
        rowcount                NUMBER(38) := NULL,
        iscompressed            BOOLEAN := FALSE,
        totalorabytes           PLS_INTEGER := NULL,
        totalinlinebytes        PLS_INTEGER := NULL,
        totaloutoflinebytes     NUMBER(38) := NULL,
        esttotalbytes           NUMBER(38) := NULL,
        esttotalbytescompressed NUMBER(38) := NULL,
        estcompressratio        BINARY_DOUBLE := 0.0,
        numcolumns              PLS_INTEGER := 0,
        numvalidcolumns         PLS_INTEGER := 0,
        metadatasize            PLS_INTEGER := 0,
        tuple                   tupledef,
        comptuple               tupledef,
        columndata              columnarray
    );
    TYPE tablearray IS TABLE of tabledef INDEX BY PLS_INTEGER;

    TYPE aquery IS RECORD
    (
        query                   CLOB,
        numresults              PLS_INTEGER,
        results                 qresult
    );

-- Constants for specific option values
--     These *must* be maintained in sync with the values defined 
--     in 'ora2ttddl.h'
    typeMapNumNone CONSTANT PLS_INTEGER := 0;
    typeMapNumStandard CONSTANT PLS_INTEGER := 1;
    typeMapNumAggressive CONSTANT PLS_INTEGER := 2;
    typeMapVarNone CONSTANT PLS_INTEGER := 0;
    typeMapVarAggressive CONSTANT PLS_INTEGER := 1;
    compressionNone CONSTANT PLS_INTEGER := 0;
    compressionFixed CONSTANT PLS_INTEGER := 1;
    compressionVariable CONSTANT PLS_INTEGER := 2;
    LOBMappingNo CONSTANT PLS_INTEGER := 0;
    LOBMappingYes CONSTANT PLS_INTEGER := 1;
    RawMappingNo CONSTANT PLS_INTEGER := 0;
    RawMappingYes CONSTANT PLS_INTEGER := 1;
    TSTZMappingNo CONSTANT PLS_INTEGER := 0;
    TSTZMappingYes CONSTANT PLS_INTEGER := 1;

-- Min and max option values
--     These *must* be maintained in sync with the values defined 
--     in 'ora2ttddl.h'
    minOnBadType CONSTANT PLS_INTEGER := 0;
    maxOnBadType CONSTANT PLS_INTEGER := 1;
    minParallel CONSTANT PLS_INTEGER := 0;
    maxParallel CONSTANT PLS_INTEGER := 128;
    minNumMapping CONSTANT PLS_INTEGER := typeMapNumNone;
    maxNumMapping CONSTANT PLS_INTEGER := typeMapNumAggressive;
    minVarMapping CONSTANT PLS_INTEGER := typeMapVarNone;
    maxVarMapping CONSTANT PLS_INTEGER := typeMapVarAggressive;
    minRawMapping CONSTANT PLS_INTEGER := RawMappingNo;
    maxRawMapping CONSTANT PLS_INTEGER := RawMappingYes;
    minLOBMapping CONSTANT PLS_INTEGER := LOBMappingNo;
    maxLOBMapping CONSTANT PLS_INTEGER := LOBMappingYes;
    minTSTZMapping CONSTANT PLS_INTEGER := TSTZMappingNo;
    maxTSTZMapping CONSTANT PLS_INTEGER := TSTZMappingYes;
    minTMPadding CONSTANT PLS_INTEGER := 0;
    maxTMPadding CONSTANT PLS_INTEGER := 1000;
    minCompression CONSTANT PLS_INTEGER := compressionNone;
    maxCompression CONSTANT PLS_INTEGER := compressionVariable;
    minCPadding CONSTANT PLS_INTEGER := 0;
    maxCPadding CONSTANT PLS_INTEGER := 1000;
    minCFactor CONSTANT PLS_INTEGER := 1;
    maxCFactor CONSTANT PLS_INTEGER := 100;
    minCMinRows CONSTANT PLS_INTEGER := 1;
    maxCMinRows CONSTANT PLS_INTEGER := 2147483647;
    minInlineLimit CONSTANT PLS_INTEGER := 0;
    maxInlineLimit CONSTANT PLS_INTEGER := 32768;

-- Default option values
--     These *must* be maintained in sync with the values defined 
--     in 'ora2ttddl.h'
    dfltOnBadType CONSTANT PLS_INTEGER := 1;
    dfltParallel CONSTANT PLS_INTEGER := NULL;
    dfltNumMapping CONSTANT PLS_INTEGER := typeMapNumStandard;
    dfltVarMapping CONSTANT PLS_INTEGER := typeMapVarNone;
    dfltRawMapping CONSTANT PLS_INTEGER := RawMappingYes;
    dfltLOBMapping CONSTANT PLS_INTEGER := LOBMappingYes;
    dfltTSTZMapping CONSTANT PLS_INTEGER := TSTZMappingYes;
    dfltTMPadding CONSTANT PLS_INTEGER := 0;
    -- dfltTMStats CONSTANT PLS_INTEGER := 0;
    dfltCompression CONSTANT PLS_INTEGER := compressionNone;
    dfltCPadding CONSTANT PLS_INTEGER := 0;
    dfltCFactor CONSTANT PLS_INTEGER := 70;
    dfltCMinRows CONSTANT PLS_INTEGER := 1024;
    -- dfltCStats CONSTANT PLS_INTEGER := 0;
    dfltInlineLimit CONSTANT PLS_INTEGER := 128;

-- Option values
    OnBadType PLS_INTEGER NOT NULL := dfltOnBadType;
    Parallel PLS_INTEGER := dfltParallel;
    NumMapping PLS_INTEGER NOT NULL := dfltNumMapping;
    VarMapping PLS_INTEGER NOT NULL := dfltVarMapping;
    RawMapping PLS_INTEGER NOT NULL := dfltRawMapping;
    LOBMapping PLS_INTEGER NOT NULL := dfltLOBMapping;
    TSTZMapping PLS_INTEGER NOT NULL := dfltTSTZMapping;
    TMPadding PLS_INTEGER NOT NULL := dfltTMPadding;
    Compression PLS_INTEGER NOT NULL := dfltCompression;
    CPadding PLS_INTEGER NOT NULL := dfltCPadding;
    CFactor PLS_INTEGER NOT NULL := dfltCFactor;
    CMinRows PLS_INTEGER NOT NULL := dfltCMinRows;
    InlineLimit PLS_INTEGER := dfltInlineLimit;

-- Other globals
    OraBytesPerChar PLS_INTEGER;
    OraBytesPerNchar PLS_INTEGER := 2;
    TableData tabledef;
    AnalysisQuery aquery;

    ttTypeAlignments aligntable :=
    aligntable(
        ttAlignUnknown,
        ttAlignInvalid,
        ttAlignVarchar2,
        ttAlignNvarchar2,
        ttAlignNumber,
        ttAlignFloat,
        ttAlignDate,
        ttAlignBinaryFloat,
        ttAlignBinaryDouble,
        ttAlignTimestamp,
        ttAlignChar,
        ttAlignNchar,
        ttAlignClob,
        ttAlignNClob,
        ttAlignBlob,
        ttAlignTinyint,
        ttAlignSmallint,
        ttAlignInteger,
        ttAlignBigint,
        ttAlignVarbinary
    );

-- Globals for RETURN values - copied to return values before return
    gWarningCount PLS_INTEGER NOT NULL := 0;
    gColumnCount PLS_INTEGER NOT NULL := 0;
    gTableInfo VARCHAR2(512) := NULL;
    gWarnings msgtable;
    gColumnData datatable;

/*
 * debugInit
 *
 * Clears down the debug table.
 */

PROCEDURE debugInit
IS
BEGIN
    IF debugMode THEN
        EXECUTE IMMEDIATE 'DELETE FROM ' || debugTableName;
        COMMIT;
    END IF;
END debugInit;

/*
 * debugMsg
 *
 * Writes a debug message to the debug table.
 */

PROCEDURE debugMsg(
                   msg IN VARCHAR2
                  )
IS
BEGIN
    IF debugMode AND msg IS NOT NULL THEN
        EXECUTE IMMEDIATE 'INSERT INTO ' || debugTableName ||
                          ' VALUES(' || debugSequenceName || '.nextval, ' ||
                          sqt || msg || sqt || ')';
        COMMIT;
    END IF;
END debugMsg;

/*
 * postWarning
 *
 *     Adds a warning message to the RETURNed warning stack.
 */

PROCEDURE postWarning(
                      warnMsg IN VARCHAR2
                     )
IS
BEGIN
    IF warnMsg IS NOT NULL THEN
        gWarningCount := gWarningCount + 1;
        gWarnings( gWarningCount ) := warnMsg;
    END IF;
END postWarning;

/*
 * isInteger
 *
 *     Checks a string to see if it is a valid positive PLS_INTEGER value
 *     in the range 0 to 999,999,999 or not.
 */

FUNCTION isInteger(
                   stringVal IN VARCHAR2
                  )
    RETURN BOOLEAN
IS
    sLen PLS_INTEGER;
    sPos PLS_INTEGER;
    sChar VARCHAR2(1);
BEGIN
    if stringVal IS NULL THEN
        RETURN FALSE;
    END IF;
    sLen := LENGTH( stringVal );
    IF sLen < 1 OR sLen > 9 THEN
        RETURN FALSE;
    END IF;
    sPos := 1;
    WHILE sPos <= sLen LOOP
        sChar := SUBSTR( stringVal, sPos, 1 );
        IF sChar < '0' OR sChar > '9' THEN
            RETURN FALSE;
        END IF;
        sPos := sPos + 1;
    END LOOP;

    RETURN TRUE;
END isInteger;

/*
 * getBytesPerChar
 *
 *     Determine the maximum number of bytes needed to hold a
 *     'character' based on a character set name.
 */

FUNCTION getBytesPerChar(
                         csname VARCHAR2
                        )
    RETURN PLS_INTEGER
IS
    bytesperchar PLS_INTEGER := 0;
    bitsperchar PLS_INTEGER := 0;
    numdigits PLS_INTEGER := 0;
    csind PLS_INTEGER := 1;
    cslen PLS_INTEGER;
    ch CHAR(1);
BEGIN
    IF csname IS NULL THEN
        GOTO fini;
    END IF;

    IF csname = 'UTF8' THEN
        bytesperchar := 4;
        GOTO fini;
    END IF;

    cslen := LENGTH(csname);

    -- skip leading non numeric characters
    ch := SUBSTR(csname, csind, 1);
    WHILE (csind <= cslen) AND ((ch < '0') OR (ch > '9')) LOOP
        csind := csind + 1;
        ch := SUBSTR(csname, csind, 1);
    END LOOP;

    -- get digits (max 2) representing bits-per-character
    WHILE (csind <= cslen) AND (numdigits < 2) AND 
          (ch >= '0') AND (ch <= '9') LOOP
        numdigits := numdigits + 1;
        bitsperchar := (10 * bitsperchar) + TO_NUMBER(ch);
        csind := csind + 1;
        ch := SUBSTR(csname, csind, 1);
    END LOOP;

    IF (csind > cslen) OR ((ch >= '0') AND (ch <= '9')) THEN
        -- badly formed character set name
        GOTO fini;
    END IF;

    CASE bitsperchar

        WHEN 7 THEN
            bytesperchar := 1;

        WHEN 8 THEN
            bytesperchar := 1;

        WHEN 16 THEN
            bytesperchar := 2;

        WHEN 32 THEN
            bytesperchar := 4;

        ELSE
            bytesperchar := 0;

    END CASE;

    <<fini>>
    RETURN bytesperchar;
END getBytesPerChar;

/*
 * checkDBCharacterSet
 *
 *     Check the DB characterset and use it to determine
 *     the number of bytes per character.
 */

FUNCTION checkDBCharacterSet(
                             errMsg OUT VARCHAR2
                            )
    RETURN PLS_INTEGER
IS
  retVal PLS_INTEGER := resultSuccess;
  csname VARCHAR2(16);
BEGIN
    SELECT VALUE INTO csname FROM NLS_DATABASE_PARAMETERS
     WHERE PARAMETER = 'NLS_CHARACTERSET';

    OraBytesPerChar := getBytesPerChar(csname);
    IF OraBytesPerChar = 0 THEN
        retVal := resultError;
        errMsg := 'Error: Unable to determine Oracle DB character size';
    END IF;

    RETURN retVal;
END checkDBCharacterSet;

/*
 * calcTTNumberStorage
 *
 *     Calculates the storage needed for a TimesTen number column.
 */

FUNCTION calcTTNumberStorage(
                             precision NUMBER,
                             scale NUMBER
                            )
    RETURN PLS_INTEGER
IS
    ttlen NUMBER;
    tscale NUMBER := scale;
BEGIN
    IF scale IS NULL THEN
        tscale := 0;
    END IF;
    IF precision IS NULL THEN
        ttlen := 22;
    ELSE
        ttlen := ROUND( ( ((precision + tscale) / 2.0) + 0.5 ) ) + 3;
        IF ttlen > 22 THEN
            ttlen := 22;
        ELSIF ttlen < 5 THEN
            ttlen := 5;
        END IF;
    END IF;
    RETURN ttlen;
END calcTTNumberStorage;

/*
 * getOraTypeText
 *
 * Create the Oracle data type text for a column definition.
 */

FUNCTION getOraTypeText(
                        col IN columndef
                       )
    RETURN VARCHAR2
IS
    typetext VARCHAR2(64) := '';
BEGIN
    CASE col.oradatatype

        WHEN oraTypeVarchar2 THEN
            typetext := 'VARCHAR2(' || TO_CHAR(col.oracharlength) || ' ';
            IF col.defaschar THEN
                typetext := typetext || 'CHAR)';
            ELSE
                typetext := typetext || 'BYTE)';
            END IF;

        WHEN oraTypeNvarchar2 THEN
            typetext := 'NVARCHAR2(' || TO_CHAR(col.oracharlength) || ')';

        WHEN oraTypeNumber THEN
            typetext := 'NUMBER';
            IF col.oraprecision IS NOT NULL THEN
                typetext := typetext || '(' || TO_CHAR(col.oraprecision);
                IF col.orascale IS NOT NULL THEN
                    typetext := typetext || ',' || TO_CHAR(col.orascale);
                END IF;
                typetext := typetext || ')';
            ELSIF col.orascale IS NOT NULL THEN
                typetext := typetext || '(' || 
                            TO_CHAR(oraNumberMaxPrecision) || ',' ||
                            TO_CHAR(col.orascale) || ')';
            END IF;

        WHEN oraTypeFloat THEN
            typetext := 'FLOAT(' || TO_CHAR(col.oraprecision) || ')';

        WHEN oraTypeDate THEN
            typetext := 'DATE';

        WHEN oraTypeBinaryFloat THEN
            typetext := 'BINARY_FLOAT';

        WHEN oraTypeBinaryDouble THEN
            typetext := 'BINARY_DOUBLE';

        WHEN oraTypeTimestamp THEN
            typetext := 'TIMESTAMP';
            IF col.orascale IS NOT NULL THEN
                typetext := typetext || '(' || TO_CHAR(col.orascale) || ')';
            END IF;

        WHEN oraTypeTimestampTZ THEN
            typetext := 'TIMESTAMP';
            IF col.ttscale IS NOT NULL THEN
                typetext := typetext || '(' || TO_CHAR(col.orascale) 
                                     || ') WITH TIME ZONE';
            END IF;

        WHEN oraTypeTimestampLTZ THEN
            typetext := 'TIMESTAMP';
            IF col.ttscale IS NOT NULL THEN
                typetext := typetext || '(' || TO_CHAR(col.orascale)
                                     || ') WITH LOCAL TIME ZONE';
            END IF;

        WHEN oraTypeChar THEN
            typetext := 'CHAR(' || TO_CHAR(col.oracharlength);
            IF col.defaschar THEN
                typetext := typetext || ' CHAR)';
            ELSE
                typetext := typetext || ' BYTE)';
            END IF;

        WHEN oraTypeNchar THEN
            typetext := 'NCHAR(' || TO_CHAR(col.oracharlength) || ')';

        WHEN oraTypeClob THEN
            typetext := 'CLOB';

        WHEN oraTypeNclob THEN
            typetext := 'NCLOB';

        WHEN oraTypeBlob THEN
            typetext := 'BLOB';

        WHEN oraTypeRaw THEN
            typetext := 'RAW(' || TO_CHAR(col.oradatalen) || ')';

        WHEN oraTypeLong THEN
            typetext := 'LONG';

        WHEN oraTypeLongraw THEN
            typetext := 'LONG RAW';

        ELSE
            typetext := 'UNKNOWN';

    END CASE;

    RETURN typetext;
END getOraTypeText;

/*
 * getTTTypeText
 *
 * Create the TimesTen data type text for a column definition.
 */

FUNCTION getTTTypeText(
                       col IN columndef
                      )
    RETURN VARCHAR2
IS
    typetext VARCHAR2(64) := '';
BEGIN
    CASE col.ttdatatype

        WHEN ttTypeVarchar2 THEN
            typetext := 'VARCHAR2(';
            IF col.oradatatype = oraTypeClob THEN
                typetext := typetext || 
                            TO_CHAR(col.ttcharlength * OraBytesPerChar) ||
                            ' BYTE)';
            ELSE
                typetext := typetext || TO_CHAR(col.ttcharlength) || ' ';
                IF col.defaschar THEN
                    typetext := typetext || 'CHAR)';
                ELSE
                    typetext := typetext || 'BYTE)';
                END IF;
            END IF;

        WHEN ttTypeNvarchar2 THEN
            typetext := 'NVARCHAR2(' || TO_CHAR(col.ttcharlength) || ')';

        WHEN ttTypeVarbinary THEN
            typetext := 'VARBINARY(' || TO_CHAR(col.ttcharlength) || ')';

        WHEN ttTypeNumber THEN
            typetext := 'NUMBER';
            IF col.ttprecision IS NOT NULL THEN
                typetext := typetext || '(' || TO_CHAR(col.ttprecision);
                IF col.ttscale IS NOT NULL THEN
                    typetext := typetext || ',' || TO_CHAR(col.ttscale);
                END IF;
                typetext := typetext || ')';
            ELSIF col.ttscale IS NOT NULL THEN
                typetext := typetext || '(' || 
                            TO_CHAR(oraNumberMaxPrecision) || ',' ||
                            TO_CHAR(col.ttscale) || ')';
            END IF;

        WHEN ttTypeFloat THEN
            typetext := 'FLOAT(' || TO_CHAR(col.ttprecision) || ')';

        WHEN ttTypeDate THEN
            typetext := 'DATE';

        WHEN ttTypeBinaryFloat THEN
            typetext := 'BINARY_FLOAT';

        WHEN ttTypeBinaryDouble THEN
            typetext := 'BINARY_DOUBLE';

        WHEN ttTypeTimestamp THEN
            typetext := 'TIMESTAMP';
            IF col.ttscale IS NOT NULL THEN
                typetext := typetext || '(' || TO_CHAR(col.ttscale) || ')';
            END IF;

        WHEN ttTypeChar THEN
            typetext := 'CHAR(' || TO_CHAR(col.ttcharlength);
            IF col.defaschar THEN
                typetext := typetext || ' CHAR)';
            ELSE
                typetext := typetext || ' BYTE)';
            END IF;

        WHEN ttTypeNchar THEN
            typetext := 'NCHAR(' || TO_CHAR(col.ttcharlength) || ')';

        WHEN ttTypeClob THEN
            typetext := 'CLOB';

        WHEN ttTypeNclob THEN
            typetext := 'NCLOB';

        WHEN ttTypeBlob THEN
            typetext := 'BLOB';

        WHEN ttTypeTinyint THEN
            typetext := 'TT_TINYINT';

        WHEN ttTypeSmallint THEN
            typetext := 'TT_SMALLINT';

        WHEN ttTypeInteger THEN
            typetext := 'TT_INTEGER';

        WHEN ttTypeBigint THEN
            typetext := 'TT_BIGINT';

        ELSE
            typetext := 'UNKNOWN';

    END CASE;

    RETURN typetext;
END getTTTypeText;

/*
 * getOraTypeCode
 *
 *     Returns our internal type code for an Oracle type string.
 */

FUNCTION getOraTypeCode(
                        oraType IN VARCHAR2
                       )
    RETURN PLS_INTEGER
IS
    retType PLS_INTEGER := oraTypeInvalid;
    ind PLS_INTEGER;
    len PLS_INTEGER := LENGTH(oraType);
    typeName VARCHAR2(64) := oraType;
    typeQualifier VARCHAR2(32) := NULL;
BEGIN
    -- extract type name and any qualifier
    ind := INSTR(oraType, '(');
    IF ind > 0 THEN
        typeName := SUBSTR(oraType,1,ind-1);
        ind := INSTR(oraType, ')');
        IF ind < 1 THEN
            GOTO fini;
        END IF;
        typeQualifier := SUBSTR(oraType,ind+1,len-ind);
    END IF;

    -- map name to internal type code
    CASE typeName

        WHEN oraTypeVarchar2S THEN
            retType := oraTypeVarchar2;

        WHEN oraTypeNvarchar2S THEN
            retType := oraTypeNvarchar2;

        WHEN oraTypeNumberS THEN
            retType := oraTypeNumber;

        WHEN oraTypeFloatS THEN
            retType := oraTypeFloat;

        WHEN oraTypeDateS THEN
            retType := oraTypeDate;

        WHEN oraTypeBinaryFloatS THEN
            retType := oraTypeBinaryFloat;

        WHEN oraTypeBinaryDoubleS THEN
            retType := oraTypeBinaryDouble;

        WHEN oraTypeTimestampS THEN
            IF typeQualifier IS NOT NULL THEN
                CASE typeQualifier
                    WHEN oraTypeWithTZ THEN
                        retType := oraTypeTimestampTZ;
                    WHEN oraTypeWithLTZ THEN
                        retType := oraTypeTimestampLTZ;
                    ELSE
                        retType := oraTypeUnknown;
                END CASE;
            ELSE
                retType := oraTypeTimestamp;
            END IF;

        WHEN oraTypeCharS THEN
            retType := oraTypeChar;

        WHEN oraTypeNcharS THEN
            retType := oraTypeNchar;

        WHEN oraTypeClobS THEN
            retType := oraTypeClob;

        WHEN oraTypeNclobS THEN
            retType := oraTypeNclob;

        WHEN oraTypeBlobS THEN
            retType := oraTypeBlob;

        WHEN oraTypeRawS THEN
            retType := oraTypeRaw;

        WHEN oraTypeLongS THEN
            retType := oraTypeLong;

        WHEN oraTypeLongrawS THEN
            retType := oraTypeLongraw;

        ELSE
            retType := oraTypeUnknown;

    END CASE;        

    <<fini>>
    RETURN retType;
END getOraTypeCode;

/*
 * getCompressibility
 *
 * Determine if an Oracle type is compressible when mapped to TimesTen
 * more specifically - whether we can run the analysis query on it
 */

FUNCTION getCompressibility(
                            oraType IN PLS_INTEGER
                           )
    RETURN BOOLEAN
IS
    retVal BOOLEAN;
BEGIN
    CASE oraType
        WHEN oraTypeVarchar2 THEN
            retVal := TRUE;
        WHEN oraTypeNvarchar2 THEN
            retVal := TRUE;
        WHEN oraTypeNumber THEN
            retVal := TRUE;
        WHEN oraTypeFloat THEN
            retVal := TRUE;
        WHEN oraTypeDate THEN
            retVal := TRUE;
        WHEN oraTypeBinaryFloat THEN
            retVal := TRUE;
        WHEN oraTypeBinaryDouble THEN
            retVal := TRUE;
        WHEN oraTypeTimestamp THEN
            retVal := TRUE;
        WHEN oraTypeTimestampTZ THEN
            retVal := TRUE;
        WHEN oraTypeTimestampLTZ THEN
            retVal := TRUE;
        WHEN oraTypeChar THEN
            retVal := TRUE;
        WHEN oraTypeNchar THEN
            retVal := TRUE;
        WHEN oraTypeRaw THEN
            retVal := TRUE;
        ELSE
            retVal := FALSE;
    END CASE;
    RETURN retVal;
END getCompressibility;

/*
 * columnMetaDataIsSane
 *
 * Check that the metatdata seems valid and set some derived values.
 */

FUNCTION columnMetaDataIsSane(
                              col IN OUT columndef
                             )
    RETURN BOOLEAN
IS
    retVal BOOLEAN := FALSE;
BEGIN
    IF col.nullable IS NULL THEN
        GOTO fini;
    END IF;

    IF col.oradatatype IS NULL THEN
        GOTO fini;
    END IF;

    CASE col.oradatatype

        WHEN oraTypeVarchar2 THEN
            IF (col.oracharlength IS NULL) OR (col.oracharlength <= 0) OR
               (col.oradatalen IS NULL) OR (col.oradatalen <= 0) OR
               (col.oracharlength > col.oradatalen) THEN
                GOTO fini;
            END IF;
            col.defaschar := (col.oradatalen > col.oracharlength);

        WHEN oraTypeNvarchar2 THEN
            IF (col.oracharlength IS NULL) OR (col.oracharlength <= 0) OR
               (col.oradatalen IS NULL) OR (col.oradatalen <= 0) OR
               (col.oracharlength > col.oradatalen) THEN
                GOTO fini;
            END IF;
            col.defaschar := (col.oradatalen > col.oracharlength);

        WHEN oraTypeChar THEN
            IF (col.oracharlength IS NULL) OR (col.oracharlength <= 0) OR
               (col.oradatalen IS NULL) OR (col.oradatalen <= 0) OR
               (col.oracharlength > col.oradatalen) THEN
                GOTO fini;
            END IF;
            col.defaschar := (col.oradatalen > col.oracharlength);

        WHEN oraTypeNchar THEN
            IF (col.oracharlength IS NULL) OR (col.oracharlength <= 0) OR
               (col.oradatalen IS NULL) OR (col.oradatalen <= 0) OR
               (col.oracharlength > col.oradatalen) THEN
                GOTO fini;
            END IF;
            col.defaschar := (col.oradatalen > col.oracharlength);

       WHEN oraTypeRaw THEN
            IF (col.oradatalen IS NULL) OR (col.oradatalen <= 0) OR
               (col.oradatalen > oraRawMaxlen) THEN
                GOTO fini;
            END IF;
            col.defaschar := FALSE;

       WHEN oraTypeNumber THEN
            IF ( (col.oraprecision IS NOT NULL) AND
                 ( (col.oraprecision < oraNumberMinPrecision) OR
                   (col.oraprecision > oraNumberMaxPrecision)  )  ) THEN
                GOTO fini;
            END IF;
            IF ( (col.orascale IS NOT NULL) AND
                 ( (col.orascale < oraNumberMinScale) OR
                   (col.orascale > oraNumberMaxScale)  )  ) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeFloat THEN
            IF  col.orascale IS NOT NULL THEN
                GOTO fini;
            END IF;
            IF ( (col.oraprecision IS NULL) OR
                 (col.oraprecision < oraFloatMinPrecision) OR
                 (col.oraprecision > oraFloatMaxPrecision)  ) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeDate THEN
            IF (col.oraprecision IS NOT NULL) OR 
               (col.orascale IS NOT NULL) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeBinaryFloat THEN
            IF (col.oraprecision IS NOT NULL) OR 
               (col.orascale IS NOT NULL) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeBinaryDouble THEN
            IF (col.oraprecision IS NOT NULL) OR 
               (col.orascale IS NOT NULL) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeTimestamp THEN
            IF  (col.orascale IS NULL) OR (col.oraprecision IS NOT NULL) THEN
                GOTO fini;
            END IF;
            IF (col.orascale < oraTimestampMinScale) OR
               (col.orascale > oraTimestampMaxScale) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeTimestampTZ THEN
            IF  (col.orascale IS NULL) OR (col.oraprecision IS NOT NULL) THEN
                GOTO fini;
            END IF;
            IF (col.orascale < oraTimestampMinScale) OR
               (col.orascale > oraTimestampMaxScale) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeTimestampLTZ THEN
            IF  (col.orascale IS NULL) OR (col.oraprecision IS NOT NULL) THEN
                GOTO fini;
            END IF;
            IF (col.orascale < oraTimestampMinScale) OR
               (col.orascale > oraTimestampMaxScale) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;

       WHEN oraTypeClob THEN
            IF  (col.orascale IS NOT NULL) OR 
                (col.oraprecision IS NOT NULL) OR
                (col.oracharlength != 0) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;
            col.oradatalen := NULL;
            col.ttdatalen := NULL;
            col.defaschar := FALSE;

       WHEN oraTypeNclob THEN
            IF  (col.orascale IS NOT NULL) OR 
                (col.oraprecision IS NOT NULL) OR
                (col.oracharlength != 0) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;
            col.oradatalen := NULL;
            col.ttdatalen := NULL;
            col.defaschar := FALSE;

       WHEN oraTypeBlob THEN
            IF  (col.orascale IS NOT NULL) OR 
                (col.oraprecision IS NOT NULL) OR
                (col.oracharlength != 0) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;
            col.oradatalen := NULL;
            col.ttdatalen := NULL;
            col.defaschar := FALSE;

       WHEN oraTypeLong THEN
            IF  (col.orascale IS NOT NULL) OR 
                (col.oraprecision IS NOT NULL) OR
                (col.oracharlength != 0) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;
            col.oradatalen := NULL;
            col.ttdatalen := NULL;
            col.defaschar := FALSE;

       WHEN oraTypeLongraw THEN
            IF  (col.orascale IS NOT NULL) OR 
                (col.oraprecision IS NOT NULL) OR
                (col.oracharlength != 0) THEN
                GOTO fini;
            END IF;
            col.oracharlength := NULL;
            col.ttcharlength := NULL;
            col.oradatalen := NULL;
            col.ttdatalen := NULL;
            col.defaschar := FALSE;

        ELSE
            GOTO fini;

    END CASE;

    retVal := TRUE;

    <<fini>>
    RETURN retVal;
END columnMetaDataIsSane;

/*
 * parseOptions
 *
 *     Parses the passed in options (AnalysisOptions) and RETURNs
 *     an error if anything is invalid otherwise it RETURNs
 *     success and the global option values have all been set
 *     accordingly.
 */

FUNCTION parseOptions(
                      optStr IN VARCHAR2,
                      errMsg OUT VARCHAR2
                     )
    RETURN PLS_INTEGER
IS
   currOpt VARCHAR2(4000);
   optName VARCHAR2(4000);
   optValS VARCHAR2(4000);
   optVal PLS_INTEGER;
   optLen PLS_INTEGER;
   oPos PLS_INTEGER := 1;
   oLen PLS_INTEGER := LENGTH(optStr);
   cPos PLS_INTEGER;
   ePos PLS_INTEGER;
BEGIN
    IF ( optStr IS NULL ) THEN
        RETURN resultSuccess;
    END IF;

    WHILE oPos <= oLen LOOP
        cPos := INSTR( optStr, ',', oPos );
        IF cPos = 0 THEN
            currOpt := SUBSTR( optStr, oPos, (oLen - oPos + 1) );
            oPos := oLen + 1;
        ELSE
            currOpt := SUBSTR( optStr, oPos, (cPos - oPos) );
            oPos := cPos + 1;
        END IF;
        optLen := LENGTH( currOpt );
        ePos := INSTR( currOpt, '=', 1 );
        IF (ePos < 2) OR (ePos = optLen) THEN
            errMsg := 'Error: Internal: Invalid option ''' || currOpt || '''';
            RETURN resultError;
        END IF;
        optName := SUBSTR( currOpt, 1, (ePos - 1) );
        optValS := SUBSTR( currOpt, (ePos + 1), (optLen - ePos) );
        IF NOT isInteger( optValS ) THEN
            errMsg := 'Error: Internal: Invalid option value ''' || currOpt || '''';
            RETURN resultError;
        END IF;
        optVal := optValS;

        CASE optName

            WHEN 'OnBadType' THEN
                IF optVal < minOnBadType OR optVal > maxOnBadType THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                OnBadType := optVal;

            WHEN 'Parallel' THEN
                IF optVal < minParallel OR optVal > maxParallel THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                Parallel := optVal;

            WHEN 'NumMapping' THEN
                IF optVal < minNumMapping OR optVal > maxNumMapping THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                NumMapping := optVal;

            WHEN 'VarMapping' THEN
                IF optVal < minVarMapping OR optVal > maxVarMapping THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                VarMapping := optVal;

            WHEN 'RawMapping' THEN
                IF optVal < minRawMapping OR optVal > maxRawMapping THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                RawMapping := optVal;

            WHEN 'LOBMapping' THEN
                IF optVal < minLOBMapping OR optVal > maxLOBMapping THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                LOBMapping := optVal;

            WHEN 'TSTZMapping' THEN
                IF optVal < minTSTZMapping OR optVal > maxTSTZMapping THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                TSTZMapping := optVal;

            WHEN 'TMPadding' THEN
                IF optVal < minTMPadding OR optVal > maxTMPadding THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                TMPadding := optVal;

            /*
            WHEN 'TMStats' THEN
                IF optVal < minTMStats OR optVal > maxTMStats THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                TMStats := optVal;
            */

            WHEN 'Compression' THEN
                IF optVal < minCompression OR optVal > maxCompression THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                Compression := optVal;

            WHEN 'CPadding' THEN
                IF optVal < minCPadding OR optVal > maxCPadding THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                CPadding := optVal;

            WHEN 'CFactor' THEN
                IF optVal < minCFactor OR optVal > maxCFactor THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                CFactor := optVal;

            WHEN 'CMinRows' THEN
                IF optVal < minCMinRows OR optVal > maxCMinRows THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                CMinRows := optVal;

            /*
            WHEN 'CStats' THEN
                IF optVal < minCStats OR optVal > maxCStats THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                CStats := optVal;
            */

            WHEN 'InlineLimit' THEN
                IF optVal < minInlineLimit OR optVal > maxInlineLimit THEN
                    errMsg := 'Error: Internal: Option value out of range ''' || currOpt || '''';
                    RETURN resultError;
                END IF;
                InlineLimit := optVal;

            ELSE
                errMsg := 'Error: Internal: Unknown option ''' || currOpt || '''';
                RETURN resultError;

        END CASE;
                
    END LOOP;

    RETURN resultSuccess;
END parseOptions;

/*
 * get RowCount
 *
 * Determines the rowcount for a table. Only used if no data analysis is
 * required.
 */

PROCEDURE getRowCount(
                      tabData IN OUT tabledef
                     )
IS
    rcquery VARCHAR2(4000);
    rowcount NUMBER := 0;
BEGIN
    rcquery := 'SELECT COUNT(*) FROM ' || qt || tabData.owner || qt ||
                      '.' || qt || tabData.name || qt || 'T1' ;
    EXECUTE IMMEDIATE rcquery INTO rowcount;
    tabData.rowcount := rowcount;
END getRowCount;

/*
 * buildAnalysisQuery
 *
 *     Builds an 'aquery' for the analysis query.
 */

FUNCTION buildAnalysisQuery(
                            tabData IN OUT tabledef,
                            thequery IN OUT aquery,
                            errMsg OUT VARCHAR2
                           )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultSuccess;
BEGIN
    thequery.query := 'SELECT ';
    IF Parallel IS NOT NULL THEN
        IF Parallel <= 1 THEN
            thequery.query := thequery.query || '/*+ NO_PARALLEL(T1) */';
        ELSE
            thequery.query := thequery.query || '/*+ PARALLEL(T1,' ||
                              TO_CHAR(Parallel) || ') */';
        END IF;
    END IF;
    thequery.query := thequery.query || ' COUNT(*)';
    thequery.numresults := 1;
    thequery.results(1) := NULL;
    
    FOR colind IN 1..tabData.numcolumns LOOP

        IF NOT tabData.columndata(colind).skip THEN

            IF (Compression > compressionNone) AND
               (tabData.columndata(colind).compressible) THEN
                thequery.query := thequery.query ||
                                  ', COUNT(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  '), COUNT(DISTINCT(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  '))';
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
            END IF; -- Compression

            IF (NumMapping > typeMapNumStandard) AND
               (tabData.columndata(colind).oradatatype = oraTypeNumber) THEN
                thequery.query := thequery.query ||
                                  ', MIN(SIGN(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  ')), MAX(LENGTH(TRUNC(ABS(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  ')))), MAX(LENGTH(ABS(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  ') - TRUNC(ABS(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  '))) - 1)';
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
            END IF; -- Numeric

            IF (VarMapping > typeMapVarNone) AND
               (((tabData.columndata(colind).oradatatype = oraTypeVarchar2) OR
                (tabData.columndata(colind).oradatatype = oraTypeNvarchar2) OR
                (tabData.columndata(colind).oradatatype = oraTypeRaw)) OR
               (((tabData.columndata(colind).oradatatype = oraTypeClob) OR
                 (tabData.columndata(colind).oradatatype = oraTypeNclob) OR
                 (tabData.columndata(colind).oradatatype = oraTypeBlob)) AND
                 (LOBMapping > LOBMappingNo))) THEN
                thequery.query := thequery.query ||
                                  ', MAX(LENGTH(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  ')), ROUND(AVG(LENGTH(' || qt ||
                                  tabData.columndata(colind).name || qt ||
                                  ')))';
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
                thequery.numresults := thequery.numresults + 1;
                thequery.results(thequery.numresults) := NULL;
            END IF; -- Varlen

        END IF; -- skip

    END LOOP;

    thequery.query := thequery.query ||
                      ' FROM ' || qt || tabData.owner || qt ||
                      '.' || qt || tabData.name || qt || 'T1' ;

    RETURN retVal;
END buildAnalysisQuery;

/*
 * executeAnalysisQuery
 *
 *     Executes the analysis query and retrieves the results.
 */

FUNCTION executeAnalysisQuery(
                              thequery IN OUT aquery,
                              errMsg OUT VARCHAR2
                             )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultError;
    acursor PLS_INTEGER;
    ignore PLS_INTEGER;
BEGIN
    BEGIN
        acursor := DBMS_SQL.OPEN_CURSOR;
        DBMS_SQL.PARSE(acursor, thequery.query, DBMS_SQL.NATIVE);
    
        FOR resind IN 1..thequery.numresults LOOP
            DBMS_SQL.DEFINE_COLUMN(acursor, resind, thequery.results(resind));
        END LOOP;

        ignore := DBMS_SQL.EXECUTE(acursor);
        IF DBMS_SQL.FETCH_ROWS(acursor) = 1 THEN
            FOR resind IN 1..thequery.numresults LOOP
                DBMS_SQL.COLUMN_VALUE(acursor, resind, 
                                      thequery.results(resind));
            END LOOP;
        ELSE
            errMsg := 'Error: Analysis query returned 0 rows';
            GOTO fini;
        END IF;
        IF DBMS_SQL.FETCH_ROWS(acursor) != 0 THEN
            errMsg := 'Error: Analysis query returned > 1 row';
            GOTO fini;
        END IF;
        
        DBMS_SQL.CLOSE_CURSOR(acursor);

        retVal := resultSuccess;
    
        EXCEPTION
            WHEN OTHERS THEN
                errMsg := 'Error: Problem executing analysis query';
                IF DBMS_SQL.IS_OPEN(acursor) THEN
                    DBMS_SQL.CLOSE_CURSOR(acursor);
                END IF;
   END;

    <<fini>>
    RETURN retVal;
END executeAnalysisQuery;

/*
 * analyseTableData
 *
 *     Perform the necessary data analysis on the Oracle table.
 */

FUNCTION analyseTableData(
                          tabData IN OUT tabledef,
                          errMsg OUT VARCHAR2
                         )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultError;
    resind PLS_INTEGER;
BEGIN
    retVal := buildAnalysisQuery( tabData, AnalysisQuery, errMsg  );
    IF retVal != resultSuccess THEN
        errMsg := 'Error: Problem building analysis query';
        GOTO fini;
    END IF;

    retVal := executeAnalysisQuery( AnalysisQuery, errMsg  );
    IF retVal != resultSuccess THEN
        GOTO fini;
    END IF;

    resind := 1;
    IF AnalysisQuery.results(resind) IS NULL THEN
        AnalysisQuery.results(resind) := 0;
    END IF;
    tabData.rowcount := AnalysisQuery.results(resind);
    resind := resind + 1;

    FOR colind IN 1..tabData.numcolumns LOOP

        IF NOT tabData.columndata(colind).skip THEN

            IF (Compression > compressionNone) AND
               (tabData.columndata(colind).compressible) THEN

                /* COUNT(colname) */
                IF AnalysisQuery.results(resind) IS NULL THEN
                    AnalysisQuery.results(resind) := 0;
                END IF;
                tabData.columndata(colind).count :=
                    AnalysisQuery.results(resind);
                resind := resind + 1;

                /* COUNT( DISTINCT(colname) ) */
                IF AnalysisQuery.results(resind) IS NULL THEN
                    tabData.columndata(colind).countdistinct :=
                        tabData.columndata(colind).count;
                ELSE
                    tabData.columndata(colind).countdistinct :=
                        AnalysisQuery.results(resind);
                END IF;

                IF tabData.columndata(colind).nullable THEN
                    tabData.columndata(colind).count :=
                        tabData.columndata(colind).count + 1;
                    tabData.columndata(colind).countdistinct :=
                        tabData.columndata(colind).countdistinct + 1;
                END IF;

                resind := resind + 1;
            END IF; -- Compression

            IF (NumMapping > typeMapNumStandard) AND
               (tabData.columndata(colind).oradatatype = oraTypeNumber) THEN

                /* MIN( SIGN(colname) ) */
                IF AnalysisQuery.results(resind) IS NULL THEN
                    tabData.columndata(colind).hasnegativevalues := FALSE;
                ELSIF AnalysisQuery.results(resind) >= 0 THEN
                    tabData.columndata(colind).hasnegativevalues := FALSE;
                ELSE
                    tabData.columndata(colind).hasnegativevalues := TRUE;
                END IF;
                resind := resind + 1;

                /* MAX( LENGTH( TRUNC( ABS(colname) ) ) ) */
                tabData.columndata(colind).maxintegrallen :=
                    AnalysisQuery.results(resind);
                resind := resind + 1;

                /* MAX( LENGTH( ABS(colname) - TRUNC( ABS(colname) ) ) - 1) */
                tabData.columndata(colind).maxfractionallen :=
                    AnalysisQuery.results(resind);
                IF (AnalysisQuery.results(resind) IS NOT NULL) AND
                   (AnalysisQuery.results(resind) > 0) THEN
                    tabData.columndata(colind).hasfractionalvalues := TRUE;
                ELSE
                    tabData.columndata(colind).hasfractionalvalues := FALSE;
                END IF;
                resind := resind + 1;

            END IF; -- Numeric

            IF (VarMapping > typeMapVarNone) AND
               (((tabData.columndata(colind).oradatatype = oraTypeVarchar2) OR
                (tabData.columndata(colind).oradatatype = oraTypeNvarchar2) OR
                (tabData.columndata(colind).oradatatype = oraTypeRaw)) OR
               (((tabData.columndata(colind).oradatatype = oraTypeClob) OR
                 (tabData.columndata(colind).oradatatype = oraTypeNclob) OR
                 (tabData.columndata(colind).oradatatype = oraTypeBlob)) AND
                 (LOBMapping > LOBMappingNo))) THEN

                /* MAX( LENGTH(colname) ) */
                tabData.columndata(colind).maxintegrallen :=
                    AnalysisQuery.results(resind);
                resind := resind + 1;

                /* ROUND( AVG( LENGTH(colname) ) )  */
                tabData.columndata(colind).avglength :=
                    AnalysisQuery.results(resind);
                resind := resind + 1;

            END IF; -- Varlen

        END IF; -- skip

    END LOOP;

    retVal := resultSuccess;

    <<fini>>
    RETURN retVal;
END analyseTableData;

/*
 * optimiseTimesTenTypes
 *
 *     Perform aggressive type mapping based on requested options
 *     and the results of the data analysis.
 */

FUNCTION optimiseTimesTenTypes(
                               tabData IN OUT tabledef,
                               errMsg OUT VARCHAR2
                              )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultSuccess;
    lenmult PLS_INTEGER;
    newlen PLS_INTEGER;
    tmpinlinelen PLS_INTEGER;
    tmp PLS_INTEGER;
    maxprecision PLS_INTEGER;
    maxscale PLS_INTEGER;
    tmpoutoflinelen NUMBER(38,0) := 0;
    padfactor BINARY_DOUBLE := 1.01 + ((TMPadding * 1.0) / 100.0);
BEGIN
    IF tabData.rowcount <= 0 THEN
        GOTO fini;
    END IF;

    FOR colind IN 1..tabData.numcolumns LOOP

        IF tabData.columndata(colind).maxfractionallen IS NULL THEN
            maxscale := 0;
        ELSE
            maxscale := tabData.columndata(colind).maxfractionallen;
        END IF;

        IF tabData.columndata(colind).maxintegrallen IS NULL THEN
            maxprecision := maxscale;
        ELSE
            maxprecision := 
                tabData.columndata(colind).maxintegrallen + maxscale;
        END IF;

        maxprecision := ROUND( maxprecision * padfactor );
        maxscale := ROUND( maxscale * padfactor );

        IF  NOT tabData.columndata(colind).skip THEN

            CASE tabData.columndata(colind).oradatatype

                WHEN oraTypeNumber THEN
                    IF NumMapping = typeMapNumAggressive THEN

                        IF (maxprecision <= oraNumberMaxPrecision) AND
                           (maxscale <= oraNumberMaxPrecision) THEN

                            tmp := tabData.columndata(colind).ttdatalen;

                            IF (tabData.columndata(colind).orascale 
                                   IS NOT NULL) AND 
                               (maxscale > 
                                    tabData.columndata(colind).orascale)
                               THEN
                                maxscale := tabData.columndata(colind).orascale;
                            END IF;

                            IF (tabData.columndata(colind).oraprecision 
                                   IS NOT NULL) AND 
                               (maxprecision > 
                                    tabData.columndata(colind).oraprecision)
                               THEN
                                maxprecision := 
                                    tabData.columndata(colind).oraprecision;
                            END IF;

                            IF NOT 
                                tabData.columndata(colind).hasfractionalvalues
                               THEN  -- integral values only

                               IF (maxprecision < 3) AND NOT
                                  tabData.columndata(colind).hasnegativevalues
                               THEN
                                   tabData.columndata(colind).wasoptimised := 
                                       TRUE;
                                   tabData.columndata(colind).ttdatatype := 
                                       ttTypeTinyint;
                                   tabData.columndata(colind).ttprecision := 2;
                                   tabData.columndata(colind).ttdatalen := 1;
                               ELSIF (maxprecision < 5) THEN
                                   tabData.columndata(colind).wasoptimised := 
                                       TRUE;
                                   tabData.columndata(colind).ttdatatype := 
                                       ttTypeSmallint;
                                   tabData.columndata(colind).ttprecision := 4;
                                   tabData.columndata(colind).ttdatalen := 2;
                               ELSIF (maxprecision < 10) THEN
                                   tabData.columndata(colind).wasoptimised := 
                                       TRUE;
                                   tabData.columndata(colind).ttdatatype := 
                                       ttTypeInteger;
                                   tabData.columndata(colind).ttprecision := 9;
                                   tabData.columndata(colind).ttdatalen := 4;
                               ELSIF (maxprecision < 19) THEN
                                   tabData.columndata(colind).wasoptimised := 
                                       TRUE;
                                   tabData.columndata(colind).ttdatatype := 
                                       ttTypeBigint;
                                   tabData.columndata(colind).ttprecision := 18;
                                   tabData.columndata(colind).ttdatalen := 8;
                               ELSE
                                   tabData.columndata(colind).wasoptimised := 
                                       TRUE;
                                   tabData.columndata(colind).ttdatatype := 
                                       ttTypeNumber;
                                   tabData.columndata(colind).ttprecision := 
                                       maxprecision;
                                   tabData.columndata(colind).ttdatalen := 
                                       calcTTNumberStorage(
                                           maxprecision,
                                           tabData.columndata(colind).ttscale);
                               END IF;

                            ELSE -- non-integral value(s)

                                tabData.columndata(colind).wasoptimised := TRUE;
                                tabData.columndata(colind).ttdatatype := 
                                    ttTypeNumber;
                                tabData.columndata(colind).ttprecision := 
                                    maxprecision;
                                tabData.columndata(colind).ttscale := 
                                    maxscale;
                                tabData.columndata(colind).ttdatalen := 
                                    calcTTNumberStorage(maxprecision,maxscale);
                            END IF;

                            IF tabData.columndata(colind).wasoptimised THEN
                                tabData.totalinlinebytes :=
                                    tabData.totalinlinebytes - tmp +
                                        tabData.columndata(colind).ttdatalen;
                            END IF;
                        END IF;
                    END IF;

            WHEN oraTypeVarchar2 THEN
                IF (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).oracharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen := 
                            tmpoutoflinelen - 
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen := 
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    IF tabData.columndata(colind).defaschar THEN
                        lenmult := 
                            tabData.columndata(colind).oradatalen /
                            tabData.columndata(colind).oracharlength;
                        tabData.columndata(colind).ttcharlength := maxprecision;
                        newlen := maxprecision * lenmult;
                    ELSE
                        lenmult := OraBytesPerChar;
                        newlen := maxprecision * lenmult;
                        IF newlen < tabData.columndata(colind).ttcharlength THEN
                            tabData.columndata(colind).ttcharlength := newlen;
                        ELSE
                            newlen := tabData.columndata(colind).ttcharlength;
                            tabData.columndata(colind).wasoptimised := FALSE;
                        END IF;
                    END IF;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;

                END IF;

            WHEN oraTypeNvarchar2 THEN
                IF (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).oracharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen :=
                            tmpoutoflinelen -
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen :=
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    IF tabData.columndata(colind).defaschar THEN
                        lenmult :=
                            tabData.columndata(colind).oradatalen /
                            tabData.columndata(colind).oracharlength;
                        tabData.columndata(colind).ttcharlength := maxprecision;
                        newlen := maxprecision * lenmult;
                    ELSE
                        lenmult := OraBytesPerChar;
                        newlen := maxprecision * lenmult;
                        IF newlen < tabData.columndata(colind).ttcharlength THEN
                            tabData.columndata(colind).ttcharlength := newlen;
                        ELSE
                            newlen := tabData.columndata(colind).ttcharlength;
                            tabData.columndata(colind).wasoptimised := FALSE;
                        END IF;
                    END IF;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;

                END IF;

            WHEN oraTypeRaw THEN
                IF (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).oracharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen :=
                            tmpoutoflinelen -
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen :=
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    newlen := maxprecision;
                    IF newlen < tabData.columndata(colind).ttcharlength THEN
                        tabData.columndata(colind).ttcharlength := newlen;
                    ELSE
                        newlen := tabData.columndata(colind).ttcharlength;
                        tabData.columndata(colind).wasoptimised := FALSE;
                    END IF;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;
                END IF;

            WHEN oraTypeLongraw THEN
                IF (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).oracharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen :=
                            tmpoutoflinelen -
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen :=
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    newlen := maxprecision;
                    IF newlen < tabData.columndata(colind).ttcharlength THEN
                        tabData.columndata(colind).ttcharlength := newlen;
                    ELSE
                        newlen := tabData.columndata(colind).ttcharlength;
                        tabData.columndata(colind).wasoptimised := FALSE;
                    END IF;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;
                END IF;

            WHEN oraTypeClob THEN
                IF (LOBMapping = LOBMappingYes) AND
                   (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).ttcharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen :=
                            tmpoutoflinelen -
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen :=
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    lenmult := OraBytesPerChar;
                    tabData.columndata(colind).ttcharlength := maxprecision;
                    newlen := maxprecision * lenmult;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;

                END IF;
               
            WHEN oraTypeNclob THEN
                IF (LOBMapping = LOBMappingYes) AND
                   (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).ttcharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen :=
                            tmpoutoflinelen -
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen :=
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    lenmult := OraBytesPerNchar;
                    tabData.columndata(colind).ttcharlength := maxprecision;
                    newlen := maxprecision * lenmult;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;

                END IF;

            WHEN oraTypeBlob THEN
                IF (LOBMapping = LOBMappingYes) AND
                   (VarMapping = typeMapVarAggressive) AND
                   (maxprecision < tabData.columndata(colind).ttcharlength) AND
                   NOT tabData.columndata(colind).alwaysoutofline THEN

                    tabData.columndata(colind).wasoptimised := TRUE;
                    IF maxprecision < 1 THEN
                        maxprecision := 1;
                    END IF;

                    tmpinlinelen := tabData.totalinlinebytes;
                    tmpoutoflinelen := tabData.totaloutoflinebytes;
                    IF tabData.columndata(colind).isoutofline THEN
                        tmpoutoflinelen :=
                            tmpoutoflinelen -
                                (tabData.columndata(colind).ttdatalen - 8);
                        tmpinlinelen := tmpinlinelen - 8;
                    ELSE
                        tmpinlinelen :=
                            tmpinlinelen - tabData.columndata(colind).ttdatalen;
                    END IF;

                    newlen := maxprecision;
                    tabData.columndata(colind).ttcharlength := maxprecision;

                    IF newlen > InlineLimit THEN
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    ELSIF (tmpinlinelen + newlen) <
                          (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := newlen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        tmpinlinelen := tmpinlinelen + newlen + 8;
                    ELSE
                        tmp := newlen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        tmpinlinelen := tmpinlinelen + 8;
                        tmpoutoflinelen := tmpoutoflinelen + tmp;
                    END IF;

                    tabData.totalinlinebytes := tmpinlinelen;
                    tabData.totaloutoflinebytes := tmpoutoflinelen;

                END IF;

            ELSE
                NULL;

            END CASE;

        END IF; -- !skip 

    END LOOP;

    retVal := resultSuccess;

    <<fini>>
    RETURN retVal;
END optimiseTimesTenTypes;

/*
 * calcTTSize
 *
 * Calculate the various size values for a TT table.
 */

PROCEDURE calcTTSize( 
                     tab IN OUT tabledef,
                     ncomptabs IN PLS_INTEGER,
                     comptabs IN OUT tablearray
                    )
IS
    totalsz NUMBER(38,0) := 0;
    totalszcomp NUMBER(38,0) := 0;
    nrows NUMBER(38,0);
    nLpages NUMBER(38,0);
    nPpages NUMBER(38,0);
    pPgSz NUMBER(38,0);
    lPgSz NUMBER(38,0);
    oolDataSz NUMBER(38,0);
    tmp NUMBER(38,0);
    tmptarray tablearray;
BEGIN
    IF tab.rowcount > 0 THEN
        nrows := tab.rowcount;
    ELSE
       GOTO fini;
    END IF;

    tmp := FLOOR( nrows / ttEntriesPerLTPg );
    IF MOD( nrows, ttEntriesPerLTPg ) != 0 THEN
        tmp := tmp + 1;
    END IF;
    nLpages := tmp;

    tmp := FLOOR( nrows / ttEntriesPerPTPg );
    IF MOD( nrows, ttEntriesPerPTPg ) != 0 THEN
        tmp := tmp + 1;
    END IF;
    nPpages := tmp;

    -- calculate sizing for non-compressed form of table
    oolDataSz := nrows * tab.tuple.outoflinesize;

    lPgSz := (ttEntriesPerLTPg * ttLTPgRowSize) + ttOverheadPerPage;

    pPgSz := (ttEntriesPerPTPg * tab.tuple.inlinesize) + ttOverheadPerPage;

    totalsz := (lPgSz * nLpages) + (pPgSz * nPpages) + oolDataSz;

    -- calculate sizing for compressed form of table
    oolDataSz := nrows * tab.comptuple.outoflinesize;

    lPgSz := (ttEntriesPerLTPg * ttLTPgRowSize) + ttOverheadPerPage;

    pPgSz := (ttEntriesPerPTPg * tab.comptuple.inlinesize) + ttOverheadPerPage;

    totalszcomp := (lPgSz * nLpages) + (pPgSz * nPpages) + oolDataSz;

    IF ncomptabs > 0 THEN
        tmptarray(1) := NULL;
        FOR ctno IN 1..ncomptabs LOOP
            calcTTSize( comptabs(ctno), 0, tmptarray );
            totalszcomp := totalszcomp + comptabs(ctno).esttotalbytes;
        END LOOP;
    END IF;

    tab.metadatasize := ttMetaPerTable + (ttMetaPerColumn * tab.numcolumns);
    tab.esttotalbytes := totalsz + tab.metadatasize;
    tab.esttotalbytescompressed := totalszcomp + tab.metadatasize;

    <<fini>>
    NULL;
END calcTTSize;

/*
 * getTTAlignment
 *
 * Returns the alignment requiremenst for a specific TimesTen type.
 */

FUNCTION getTTAlignment(
                        ttType IN PLS_INTEGER
                       )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := 0;
BEGIN
    IF (ttType >= ttTypeMin) AND (ttType <= ttTypeMax) THEN
        retVal := ttTypeAlignments(ttType + 1);
    END IF;

    RETURN retVal;
END getTTAlignment;

/*
 * sortColsByAlignment
 *
 * Sort a list of columns into order of decreasing alignment requirements.
 */

PROCEDURE sortColsByAlignment(
                              ncols IN PLS_INTEGER,
                              cols IN OUT columnarray
                             )
IS
    swapped BOOLEAN := TRUE;
    i PLS_INTEGER := 1;
    tmp columndef;
BEGIN
    WHILE swapped AND (i <= ncols) LOOP
        swapped := FALSE;
        FOR j IN 1..(ncols-1) LOOP
            IF cols(j).ttalignment < cols(j+1).ttalignment THEN
                tmp := cols(j);
                cols(j) := cols(j+1);
                cols(j+1) := tmp;
                swapped := TRUE;
            END IF;
        END LOOP;
        i := i + 1;
    END LOOP;
END sortColsByAlignment;

/*
 *  TTInlineSz
 *
 *  Calculate inline size of a tuple.
 */

FUNCTION TTInlineSz(
                    col IN columndef,
                    compressed IN BOOLEAN
                   )
    RETURN PLS_INTEGER
IS
    sz PLS_INTEGER := 0;
BEGIN
    CASE

        WHEN col.ttdatatype = ttTypeTinyint OR
             col.ttdatatype = ttTypeSmallint OR
             col.ttdatatype = ttTypeInteger OR
             col.ttdatatype = ttTypeBigint OR
             col.ttdatatype = ttTypeNumber OR
             col.ttdatatype = ttTypeFloat OR
             col.ttdatatype = ttTypeDate OR
             col.ttdatatype = ttTypeBinaryFloat OR
             col.ttdatatype = ttTypeBinaryDouble OR
             col.ttdatatype = ttTypeTimestamp OR
             col.ttdatatype = ttTypeChar OR
             col.ttdatatype = ttTypeNchar THEN

             IF compressed AND (col.ttptrsize > 0) THEN
                 sz := col.ttptrsize;
             ELSE
                 sz := col.ttdatalen;
             END IF;

        WHEN col.ttdatatype = ttTypeVarchar2 OR
             col.ttdatatype = ttTypeNvarchar2 OR
             col.ttdatatype = ttTypeVarbinary THEN

             IF compressed AND (col.ttptrsize > 0) THEN
                 sz := col.ttptrsize;
             ELSIF col.isoutofline THEN
                 sz := 8;
             ELSE
                 sz := col.ttdatalen;
             END IF;
    
        WHEN col.ttdatatype = ttTypeClob OR
             col.ttdatatype = ttTypeNClob OR
             col.ttdatatype = ttTypeBlob THEN

             sz := 8;

    END CASE;

    RETURN sz;
END TTInlineSz;

/*
 * TTOutOfLineSz
 *
 * Calculates the out of line size of a tuple.
 */

FUNCTION TTOutOfLineSz(
                       col IN columndef,
                       compressed IN BOOLEAN,
                       frac IN NUMBER
                      )
    RETURN NUMBER
IS
    sz NUMBER := 0;
    tmp NUMBER;
BEGIN
    CASE

        WHEN col.ttdatatype = ttTypeTinyint OR
             col.ttdatatype = ttTypeSmallint OR
             col.ttdatatype = ttTypeInteger OR
             col.ttdatatype = ttTypeBigint OR
             col.ttdatatype = ttTypeNumber OR
             col.ttdatatype = ttTypeFloat OR
             col.ttdatatype = ttTypeDate OR
             col.ttdatatype = ttTypeBinaryFloat OR
             col.ttdatatype = ttTypeBinaryDouble OR
             col.ttdatatype = ttTypeTimestamp OR
             col.ttdatatype = ttTypeChar OR
             col.ttdatatype = ttTypeNchar THEN

             sz := 0;

        WHEN col.ttdatatype = ttTypeVarchar2 OR
             col.ttdatatype = ttTypeNvarchar2 OR
             col.ttdatatype = ttTypeVarbinary THEN

             IF compressed AND (col.ttptrsize > 0) THEN
                 sz := 0;
             ELSIF col.isoutofline THEN
                 IF col.avglength IS NOT NULL THEN
                     sz := col.avglength + 16;
                 ELSE
                     tmp := col.ttdatalen - 24;
                     sz := ROUND( (tmp * frac) + 0.5 ) + 16;
                 END IF;
             ELSE
                 sz := 0;
             END IF;
    

        WHEN col.ttdatatype = ttTypeClob OR
             col.ttdatatype = ttTypeNClob THEN

             IF col.avglength IS NOT NULL THEN
                 sz := col.avglength + 16;
             ELSE
                 sz := ROUND( ( 4 * 1024 * 1024 * frac) + 0.5 ) + 16;
             END IF;
             IF sz < 32 THEN
                 sz := 32;
             END IF;

        WHEN col.ttdatatype = ttTypeBlob THEN

             IF col.avglength IS NOT NULL THEN
                 sz := col.avglength + 16;
             ELSE
                 sz := ROUND( ( 16 * 1024 * 1024 * frac) + 0.5 ) + 16;
             END IF;
             IF sz < 32 THEN
                 sz := 32;
             END IF;

    END CASE;

    RETURN sz;
END TTOutOfLineSz;

/*
 * calculateTTTupleSize
 *
 * Computes the size of a TT tuple.
 */

PROCEDURE calculateTTTupleSize(
                               tuple IN OUT tupledef,
                               varfraction IN NUMBER,
                               compressed IN BOOLEAN,
                               inlinesz IN OUT PLS_INTEGER,
                               outoflinesz IN OUT NUMBER
                              )
IS
    tupoffset PLS_INTEGER := 0;
    tmp PLS_INTEGER;
BEGIN
    inlinesz := 0;
    outoflinesz := 0;

    FOR colno IN 1..tuple.numcols LOOP

        tmp := MOD( tupoffset, tuple.cols(colno).ttalignment );
        IF tmp != 0 THEN
            tupoffset := tupoffset + (tuple.cols(colno).ttalignment - tmp);
        END IF;

        tupoffset := tupoffset + TTInlineSz( tuple.cols(colno), compressed );
        outoflinesz := 
            outoflinesz + 
                TTOutOfLineSz( tuple.cols(colno), compressed, varfraction );
    END LOOP;

    tmp := MOD( tupoffset, ttPTPgTupAlign );
    IF tmp != 0 THEN
        tupoffset := tupoffset + (ttPTPgTupAlign - tmp);
    END IF;

    inlinesz := tupoffset;
END calculateTTTupleSize;

/*
 * buildTuple
 *
 * Build a TimesTen format tuple structure for a table.
 */

FUNCTION buildTuple(
                    tuple IN OUT tupledef,
                    ncols IN PLS_INTEGER,
                    cols IN columnarray,
                    compressed IN BOOLEAN,
                    errMsg OUT VARCHAR2
                   )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultError;
    colNulls columndef;
    colSlot columndef;
    colRefcnt columndef;
    colind PLS_INTEGER;
    colpos PLS_INTEGER;
    numcols PLS_INTEGER;
    inlinesz PLS_INTEGER := 0;
    outoflinesz NUMBER(38,0) := 0;
    tmptup tupledef;
BEGIN
    
    IF ncols < 1 THEN
        errMsg := 'Error: table has empty column list';
        GOTO fini;
    END IF;

    errMsg := NULL;

    colNulls.ttdatatype := ttTypeChar;
    colNulls.ttdatalen := 1;
    colNulls.name := 'NULLS';

    colSlot.ttdatatype := ttTypeSmallint;
    colSlot.ttdatalen := 2;
    colSlot.name := 'SLOTNO';

    colRefcnt.ttdatatype := ttTypeInteger;
    colRefcnt.ttdatalen := 4;
    colRefcnt.name := 'REFCNT';

    colind := 1;
    colpos := 1;

    WHILE colind <= ncols LOOP

        IF NOT cols(colind).skip THEN

            tmptup.cols(colpos) := cols(colind);

            IF cols(colind).nullable THEN
                tmptup.numnullable := tmptup.numnullable + 1;
            END IF;

            IF compressed AND (cols(colind).ttptrsize > 0) THEN
                tmptup.cols(colpos).ttalignment := cols(colind).ttptrsize;
            ELSE
                tmptup.cols(colpos).ttalignment := 
                    getTTAlignment( cols(colind).ttdatatype );
            END IF;

            colpos := colpos + 1;

        END IF;

        colind := colind + 1;
    END LOOP;

    IF colpos < 2 THEN
        errMsg := 'Error: table has no valid columns';
        GOTO fini;
    END IF;
    numcols := colpos - 1;

    sortColsByAlignment( numcols, tmptup.cols );

    -- add NULL indicator bytes if required
    IF tmptup.numnullable > 0 THEN

        colNulls.ttdatalen := tmptup.numnullable / (8*ttPTPgBitsPerNull);
        IF MOD(tmptup.numnullable,(8*ttPTPgBitsPerNull)) > 0 THEN
            colNulls.ttdatalen := colNulls.ttdatalen + 1;
        END IF;
        colNulls.ttalignment := getTTAlignment( colNulls.ttdatatype );
        tmptup.cols(colpos) := colNulls;
        colpos := colpos + 1;
    END IF;

    -- add slotno and refcnt

    colSlot.ttdatalen := ttPTPgSlotnoAlign;
    colSlot.ttalignment := ttPTPgSlotnoAlign;
    tmptup.cols(colpos) := colSlot;
    colpos := colpos + 1;

    colRefcnt.ttdatalen := ttPTPgRefcntAlign;
    colRefcnt.ttalignment := ttPTPgRefcntAlign;
    tmptup.cols(colpos) := colRefcnt;
    colpos := colpos + 1;

    numcols := colpos - 1;
    tmptup.numcols := numcols;

    calculateTTTupleSize( tmptup, ttDfltVarlenUsage, compressed,
                          inlinesz, outoflinesz );
    tmptup.inlinesize := inlinesz;
    -- tmptup.inlinesize := inlinesz + 4;
    tmptup.outoflinesize := outoflinesz;
    
    tuple := tmptup;
    retVal := resultSuccess;

    <<fini>>
    RETURN retVal;
END buildTuple;

/*
 * createCompTable
 *
 * Create a table representing a compressed column.
 */

FUNCTION createCompTable(
                         column IN OUT columndef,
                         ctable IN OUT tabledef,
                         errMsg OUT VARCHAR2
                        )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultSuccess;
    tuple tupledef;
BEGIN
    ctable.owner := NULL;
    ctable.name := column.name;

    ctable.numcolumns := 1;
    ctable.numvalidcolumns := 1;
    ctable.columndata(1) := column;
    ctable.columndata(1).ttptrsize := 0;
    ctable.columndata(1).nullable := FALSE;
    ctable.rowcount := column.countdistinct;

    retVal := buildTuple( tuple,
                          ctable.numcolumns,
                          ctable.columndata,
                          FALSE,
                          errMsg );
    IF retVal != resultSuccess THEN
        GOTO fini;
    END IF;

    ctable.tuple := tuple;
    ctable.comptuple := tuple;

    <<fini>>
    RETURN retVal;
END createCompTable;
                         
/*
 * computeTTTableSize
 *
 * Computes the expected size of a table in TimesTen for both
 * the uncompressed and compressed forms.
 */

FUNCTION computeTTTableSize(
                            tabData IN OUT tabledef,
                            errMsg OUT VARCHAR2
                           )
    RETURN PLS_INTEGER
IS
    retVal  PLS_INTEGER := resultSuccess;
    colno   PLS_INTEGER;
    tabind  PLS_INTEGER;
    col     columndef;
    tuple   tupledef;
    comptab tabledef;
    comptables tablearray;
BEGIN

    errMsg := NULL;
    tabind := 1;

    FOR colno IN 1..tabData.numcolumns LOOP

        IF tabData.columndata(colno).ttptrsize > 0 THEN

            compTables(tabind) := NULL;
            retVal := createCompTable( tabData.columndata(colno),
                                       comptables(tabind),
                                       errMsg );
            IF retVal != resultSuccess THEN
                GOTO fini;
            END IF;
            tabind := tabind + 1;

        END IF;

    END LOOP;
    tabind := tabind - 1;
    
    -- build uncompressed tuple
    retVal := buildTuple( tuple,
                          tabData.numcolumns,
                          tabData.columndata,
                          FALSE,
                          errMsg );
    IF retVal != resultSuccess THEN
        GOTO fini;
    END IF;
    tabData.tuple := tuple;

    -- build compressed tuple
    retVal := buildTuple( tuple,
                          tabData.numcolumns,
                          tabData.columndata,
                          TRUE,
                          errMsg );
    IF retVal != resultSuccess THEN
        GOTO fini;
    END IF;
    tabData.comptuple := tuple;

    -- calculate the various sizes
    calcTTSize( tabData,
                tabind,
                comptables );

    <<fini>>
    RETURN retVal;
END computeTTTableSize;

/*
 * evaluateCompression
 *
 *     Perform compression evaluation based on requested options
 *     and the results of the data analysis. Also computes
 *     table sizing.
 */

FUNCTION evaluateCompression(
                             tabData IN OUT tabledef,
                             errMsg OUT VARCHAR2
                            )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultSuccess;
    ttsznocompress NUMBER(38,0);
    ttszcompress NUMBER(38,0);
    nrows NUMBER(38,0);
    ndistinct NUMBER(38,0);
    ttptrsize PLS_INTEGER;
    numcompressedcols PLS_INTEGER;
    tmp1 BINARY_DOUBLE;
    tmp2 BINARY_DOUBLE;
    compressionfactor BINARY_DOUBLE;
    targetcompressionfactor BINARY_DOUBLE;
    padfactor BINARY_DOUBLE;
    totsznocompress BINARY_DOUBLE;
    totszcompress BINARY_DOUBLE;
    colno PLS_INTEGER;
BEGIN
    errMsg := NULL;
    ttsznocompress := 0.0;
    ttszcompress := 0.0;
    totsznocompress := 0.0;
    totszcompress := 0.0;
    numcompressedcols := 0;
    padfactor := CPadding;
    padfactor := 1.0 + (padfactor / 100.0);
    targetcompressionfactor := CFactor;
    targetcompressionfactor := targetcompressionfactor / 100.0;

    IF tabData.rowcount > 0 AND Compression > compressionNone THEN

        FOR colno IN 1..tabData.numcolumns LOOP

            IF NOT tabData.columndata(colno).skip THEN

                ndistinct := tabData.columndata(colno).countdistinct;
                IF ndistinct <= 0 THEN -- assume worst case
                    ndistinct := tabData.rowcount;
                END IF;
                IF CPadding > 0 THEN
                    ndistinct := ROUND( ndistinct * padfactor );
                END IF;

                IF Compression = compressionFixed THEN -- COMPRESS_FIXED
                    IF ndistinct <= 4294967295 THEN
                        ttptrsize := 4;
                    ELSE
                        ttptrsize := 0;
                    END IF;

                ELSE -- COMPRESS_VARIABLE

                    IF ndistinct <= 255 THEN
                        ttptrsize := 1;
                    ELSIF ndistinct <= 65535 THEN
                        ttptrsize := 2;
                    ELSIF ndistinct <= 4294967295 THEN
                        ttptrsize := 4;
                    ELSE
                        ttptrsize := 0;
                    END IF;

                END IF;

                IF ttptrsize > 0 THEN
                    ttsznocompress := 
                          tabData.rowcount *
                          tabData.columndata(colno).ttdatalen;
                    ttszcompress := 
                        ( ndistinct *
                          tabData.columndata(colno).ttdatalen ) +
                        ( tabData.rowcount * ttptrsize );
                    tmp1 := ttszcompress;
                    tmp2 := ttsznocompress;
                    compressionfactor := tmp1 / tmp2;
                    IF compressionfactor > targetcompressionfactor THEN
                        ttptrsize := 0;
                        totsznocompress := totsznocompress + ttsznocompress;
                        totszcompress := totszcompress + ttsznocompress;
                    ELSE
                        tabData.columndata(colno).estcompressratio :=
                            compressionfactor;
                        totsznocompress := totsznocompress + ttsznocompress;
                        totszcompress := totszcompress + ttszcompress;
                    END IF;

                ELSE -- ttptrsize = 0
                    ttsznocompress := 
                          tabData.rowcount *
                          tabData.columndata(colno).ttdatalen;
                    totsznocompress := totsznocompress + ttsznocompress;
                    totszcompress := totszcompress + ttsznocompress;

                END IF; -- ttptrsize = 0

                tabData.columndata(colno).ttptrsize := ttptrsize;
                IF ttptrsize > 0 THEN
                    numcompressedcols := numcompressedcols + 1;
                END IF;

            END IF; -- skip
        END LOOP;

        retVal := computeTTTableSize( tabData, errMsg );
        IF retVal != resultSuccess THEN
            GOTO fini;
        END IF;

        IF (tabData.esttotalbytes > 0) AND
           (tabData.esttotalbytescompressed > 0) THEN
            tmp1 := tabData.esttotalbytescompressed;
            tmp2 := tabData.esttotalbytes;
            tabData.estcompressratio := tmp1 / tmp2;
            IF (tabData.rowcount >= CMinRows) AND
               (tabData.estcompressratio <= targetcompressionfactor) AND 
               (numcompressedcols > 0) THEN
                tabData.iscompressed := TRUE;
            END IF;
        END IF;

    ELSE -- no compression evaluation

        IF tabData.rowcount > 0 THEN
            retVal := computeTTTableSize( tabData, errMsg );
            IF retVal != resultSuccess THEN
                GOTO fini;
            END IF;
        END IF;

    END IF; -- no compression evaluation

    <<fini>>
    RETURN retVal;
END evaluateCompression;

/*
 * computeTimesTenTableSize
 *
 *     Computes table size in TimesTen.
 *     Used when no compression analysis was done.
 */

FUNCTION computeTimesTenTableSize(
                                  tabData IN OUT tabledef,
                                  errMsg OUT VARCHAR2
                                 )
    RETURN PLS_INTEGER
IS
BEGIN
    IF tabData.rowcount > 0 THEN
        RETURN computeTTTableSize( tabData, errMsg );
    ELSE
        RETURN resultSuccess;
    END IF;
END computeTimesTenTableSize;

/*
 * getColumnInfo
 *
 *     Retrieves all necessary info about a table's columns.
 */

FUNCTION getColumnInfo(
                          tabData IN OUT tabledef,
                          errMsg OUT VARCHAR2
                      )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultError;
    foundBadType BOOLEAN := FALSE;
    colTypeIsBad BOOLEAN := FALSE;
    oraType PLS_INTEGER;
    warnMsg VARCHAR2(512);
    colName VARCHAR2(30);
    dataType VARCHAR2(108);
    charColDeclLength PLS_INTEGER;
    charLength PLS_INTEGER;
    dataPrecision PLS_INTEGER;
    dataScale PLS_INTEGER;
    dataLength PLS_INTEGER;
    nullAble VARCHAR2(2);
    CURSOR colInfo (oname VARCHAR2, tname VARCHAR2) IS
        SELECT COLUMN_NAME, DATA_TYPE, CHAR_COL_DECL_LENGTH, CHAR_LENGTH, 
               DATA_PRECISION, DATA_SCALE, DATA_LENGTH, NULLABLE 
          FROM ALL_TAB_COLUMNS WHERE OWNER = oname AND TABLE_NAME = tname 
          ORDER BY COLUMN_ID;
BEGIN
    errMsg := NULL;

    OPEN colInfo( tabData.owner, tabData.name );
    LOOP
        colTypeIsBad := FALSE;
        FETCH colInfo INTO colName, dataType, charColDeclLength, charLength,
                           dataPrecision, dataScale, dataLength, nullAble;
        EXIT WHEN colInfo%NOTFOUND;

        IF dataType IS NULL THEN
            errMsg := 'Error: NULL RETURNed for ''data_type'' for ''' ||
                      tabData.owner || '.' || tabData.name || '.' ||
                      colName || '''';
            CLOSE colInfo;
            GOTO fini;
        END IF;

        -- add data to record
        tabData.numcolumns := tabData.numcolumns + 1;
        tabData.columndata(tabData.numColumns).name := colName;
        tabData.columndata(tabData.numColumns).oratypetext := dataType;
        tabData.columndata(tabData.numColumns).oradatalen := dataLength;
        oraType := getOraTypeCode(dataType);
        tabData.columndata(tabData.numColumns).oradatatype := oraType;
        tabData.columndata(tabData.numColumns).compressible := 
            getCompressibility(tabData.columndata(tabData.numColumns).oradatatype);

        -- Type specific checks and actions
        IF oraType = oraTypeUnknown OR oraType = oraTypeInvalid THEN
            foundBadType := TRUE;
            colTypeIsBad := TRUE;
            warnMsg := 'Warning: Unsupported data type ''' || dataType || 
                       ''' for ''' || tabData.owner || '.' || tabData.name || 
                       '.' || colName || '''';
            IF OnBadType > 0 THEN
                warnMsg := warnMsg || ' - skipping column';
            END IF;
            postWarning(warnMsg);
        ELSIF oraType = oraTypeTimestampTZ THEN
            IF TSTZMapping > TSTZMappingNo THEN
                warnMsg := 'Warning: Column ''' || tabData.owner || '.' || 
                           tabData.name || '.' || colName || ''' has type ' ||
                           'TIMESTAMP WITH TIMEZONE - timezone information ' ||
                           'will be lost';
                postWarning(warnMsg);
            ELSE
                foundBadType := TRUE;
                colTypeIsBad := TRUE;
                warnMsg := 'Warning: Column ''' || tabData.owner || '.' || 
                           tabData.name || '.' || colName || ''' has type ' ||
                           'TIMESTAMP WITH TIMEZONE but mapping not requested';
                IF OnBadType > 0 THEN
                    warnMsg := warnMsg || ' - skipping column';
                END IF;
                postWarning(warnMsg);
            END IF;
        ELSIF (RawMapping = RawMappingNo) AND
              ( (oraType = oraTypeRaw) OR (oraType = oraTypeLong) OR
                (oraType = oraTypeLongraw) ) THEN
            foundBadType := TRUE;
            colTypeIsBad := TRUE;
            warnMsg := 'Warning: Column ''' || tabData.owner || '.' || 
                       tabData.name || '.' || colName || ''' has type ' ||
                       dataType || ' but mapping not requested';
            IF OnBadType > 0 THEN
                warnMsg := warnMsg || ' - skipping column';
            END IF;
            postWarning(warnMsg);
        END IF;

        IF colTypeIsBad THEN
            tabData.columndata(tabData.numColumns).skip := TRUE;
        ELSE
            tabData.numvalidcolumns := tabData.numvalidcolumns + 1;
            tabData.columndata(tabData.numColumns).skip := FALSE;

            IF nullAble = 'Y' THEN
                tabData.columndata(tabData.numColumns).nullable := TRUE;
            ELSE
                tabData.columndata(tabData.numColumns).nullable := FALSE;
            END IF;

            tabData.columndata(tabData.numColumns).oracharlength := 
                charLength;
            tabData.columndata(tabData.numColumns).ttcharlength := 
                charLength;
            
            tabData.columndata(tabData.numColumns).oraprecision := 
                dataPrecision;
            tabData.columndata(tabData.numColumns).ttprecision := 
                dataPrecision;
            
            tabData.columndata(tabData.numColumns).orascale := 
                dataScale;
            tabData.columndata(tabData.numColumns).ttscale := 
                dataScale;
            
            IF (dataScale IS NULL) OR (dataScale > 0) THEN
                tabData.columndata(tabData.numColumns).hasfractionalvalues := 
                    TRUE;
            END IF;

            IF NOT 
               columnMetaDataIsSane( tabData.columndata(tabData.numColumns) ) 
               THEN
                errMsg := 'Error: Inconsistent metadata retrieved for ''' ||
                      tabData.owner || '.' || tabData.name || '.' ||
                      colName || '''';
                CLOSE colInfo;
                GOTO fini;
            END IF;

        END IF;
        
    END LOOP;

    CLOSE colInfo;

    IF foundBadType THEN
        IF OnBadType > 0 THEN
            IF tabData.numvalidcolumns > 0 THEN
                retVal := resultSuccess;
            ELSE
                retVal := resultSkipTable;
                warnMsg := 'Warning: Skipping table ''' || tabData.owner || 
                           '.' || tabData.name || ''' - no valid columns';
                postWarning(warnMsg);
            END IF;
        ELSE
            retVal := resultSkipTable;
            warnMsg := 'Warning: Skipping table ''' || tabData.owner || '.' || 
                       tabData.name || ''' due to unsupported data type(s)';
            postWarning(warnMsg);
        END IF;
    ELSE
        retVal := resultSuccess;
    END IF;

    <<fini>>
    RETURN retVal;
END getColumnInfo;

/*
 * assignTimesTenTypes
 *
 *     Performs initial type assignments.
 */

FUNCTION assignTimesTenTypes(
                             tabData IN OUT tabledef,
                             errMsg OUT VARCHAR2
                            )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultError;
    scale PLS_INTEGER;
    tmp PLS_INTEGER;
    ftmp NUMBER;
    totoralen PLS_INTEGER := 0;
    totinlinelen PLS_INTEGER := 0;
    totoutoflinelen NUMBER(38,0) := 0;
BEGIN
    FOR colind IN 1..tabData.numcolumns LOOP

        IF  NOT tabData.columndata(colind).skip THEN 

        CASE tabData.columndata(colind).oradatatype

            WHEN oraTypeNumber THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                IF (NumMapping = typeMapNumNone) OR
                   (tabData.columndata(colind).orascale != 0) OR
                   (tabData.columndata(colind).oraprecision IS NULL) THEN
                    tabData.columndata(colind).ttdatatype := ttTypeNumber;
                    tabData.columndata(colind).ttprecision := 
                        tabData.columndata(colind).oraprecision;
                    tabData.columndata(colind).ttscale := 
                        tabData.columndata(colind).orascale;
                ELSE
                    IF tabData.columndata(colind).oraprecision < 5 THEN
                        tabData.columndata(colind).wasoptimised := TRUE;
                        tabData.columndata(colind).ttdatatype := ttTypeSmallint;
                        tabData.columndata(colind).ttdatalen := 2;
                        tabData.columndata(colind).ttprecision := 4;
                    ELSIF tabData.columndata(colind).oraprecision < 10 THEN
                        tabData.columndata(colind).wasoptimised := TRUE;
                        tabData.columndata(colind).ttdatatype := ttTypeInteger;
                        tabData.columndata(colind).ttdatalen := 4;
                        tabData.columndata(colind).ttprecision := 9;
                    ELSIF tabData.columndata(colind).oraprecision < 19 THEN
                        tabData.columndata(colind).wasoptimised := TRUE;
                        tabData.columndata(colind).ttdatatype := ttTypeBigint;
                        tabData.columndata(colind).ttdatalen := 8;
                        tabData.columndata(colind).ttprecision := 18;
                    ELSE
                        tabData.columndata(colind).ttdatatype := ttTypeNumber;
                    END IF;
                END IF;
                IF tabData.columndata(colind).ttdatatype = ttTypeNumber THEN
                    tabData.columndata(colind).ttdatalen := 
                    calcTTNumberStorage( tabData.columndata(colind).ttprecision,
                                         tabData.columndata(colind).ttscale );
                END IF;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;

            WHEN oraTypeVarchar2 THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeVarchar2;
                IF tabData.columndata(colind).oradatalen > InlineLimit THEN
                    tmp := tabData.columndata(colind).oradatalen + 16;
                    IF tmp < 32 THEN
                        tmp := 32;
                    END IF;
                    tabData.columndata(colind).ttdatalen := tmp + 8;
                    tabData.columndata(colind).isoutofline := TRUE;
                    totinlinelen := totinlinelen + 8;
                    totoutoflinelen := totoutoflinelen + tmp;
                ELSIF (totinlinelen + tabData.columndata(colind).oradatalen + 8)
                          < (ttMaxInlineLimit - 10) THEN
                    tabData.columndata(colind).ttdatalen := 
                        tabData.columndata(colind).oradatalen + 8;
                    tabData.columndata(colind).isoutofline := FALSE;
                    totinlinelen := totinlinelen + 
                        tabData.columndata(colind).ttdatalen;
                ELSE
                    tmp := tabData.columndata(colind).oradatalen + 16;
                    IF tmp < 32 THEN
                        tmp := 32;
                    END IF;
                    tabData.columndata(colind).ttdatalen := tmp + 8;
                    tabData.columndata(colind).isoutofline := TRUE;
                    totinlinelen := totinlinelen + 8;
                    totoutoflinelen := totoutoflinelen + tmp;
                END IF;

            WHEN oraTypeRaw THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeVarbinary;
                tabData.columndata(colind).ttcharlength := 
                    tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).oracharlength := 
                    tabData.columndata(colind).ttcharlength;
                tabData.columndata(colind).defaschar := FALSE;
                IF tabData.columndata(colind).oradatalen > InlineLimit THEN
                    tmp := tabData.columndata(colind).oradatalen + 16;
                    IF tmp < 32 THEN
                        tmp := 32;
                    END IF;
                    tabData.columndata(colind).ttdatalen := tmp + 8;
                    tabData.columndata(colind).isoutofline := TRUE;
                    totinlinelen := totinlinelen + 8;
                    totoutoflinelen := totoutoflinelen + tmp;
                ELSIF (totinlinelen + tabData.columndata(colind).oradatalen + 8)
                          < (ttMaxInlineLimit - 10) THEN
                    tabData.columndata(colind).ttdatalen := 
                        tabData.columndata(colind).oradatalen + 8;
                    tabData.columndata(colind).isoutofline := FALSE;
                    totinlinelen := totinlinelen + 
                                    tabData.columndata(colind).ttdatalen;
                ELSE
                    tmp := tabData.columndata(colind).oradatalen + 16;
                    IF tmp < 32 THEN
                        tmp := 32;
                    END IF;
                    tabData.columndata(colind).ttdatalen := tmp + 8;
                    tabData.columndata(colind).isoutofline := TRUE;
                    totinlinelen := totinlinelen + 8;
                    totoutoflinelen := totoutoflinelen + tmp;
                END IF;

            WHEN oraTypeLong THEN
                tabData.columndata(colind).ttdatatype := ttTypeVarchar2;
                tabData.columndata(colind).oracharlength := ttVarMaxlen;
                tabData.columndata(colind).ttcharlength := ttVarMaxlen;
                tabData.columndata(colind).alwaysoutofline := TRUE;
                tabData.columndata(colind).isoutofline := TRUE;
                tmp := ttVarMaxlen + 16;
                tabData.columndata(colind).ttdatalen := tmp + 8;
                totinlinelen := totinlinelen + 8;
                totoutoflinelen := totoutoflinelen + tmp;

            WHEN oraTypeLongraw THEN
                tabData.columndata(colind).ttdatatype := ttTypeVarbinary;
                tabData.columndata(colind).oracharlength := ttVarMaxlen;
                tabData.columndata(colind).ttcharlength := ttVarMaxlen;
                tabData.columndata(colind).alwaysoutofline := TRUE;
                tabData.columndata(colind).isoutofline := TRUE;
                tmp := ttVarMaxlen + 16;
                tabData.columndata(colind).ttdatalen := tmp + 8;
                totinlinelen := totinlinelen + 8;
                totoutoflinelen := totoutoflinelen + tmp;

            WHEN oraTypeNvarchar2 THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeNvarchar2;
                IF tabData.columndata(colind).oradatalen > InlineLimit THEN
                    tmp := tabData.columndata(colind).oradatalen + 16;
                    IF tmp < 32 THEN
                        tmp := 32;
                    END IF;
                    tabData.columndata(colind).ttdatalen := tmp + 8;
                    tabData.columndata(colind).isoutofline := TRUE;
                    totinlinelen := totinlinelen + 8;
                    totoutoflinelen := totoutoflinelen + tmp;
                ELSIF (totinlinelen + tabData.columndata(colind).oradatalen + 8)
                          < (ttMaxInlineLimit - 10) THEN
                    tabData.columndata(colind).ttdatalen := 
                        tabData.columndata(colind).oradatalen + 8;
                    tabData.columndata(colind).isoutofline := FALSE;
                    totinlinelen := totinlinelen + 
                                    tabData.columndata(colind).ttdatalen;
                ELSE
                    tmp := tabData.columndata(colind).oradatalen + 16;
                    IF tmp < 32 THEN
                        tmp := 32;
                    END IF;
                    tabData.columndata(colind).ttdatalen := tmp + 8;
                    tabData.columndata(colind).isoutofline := TRUE;
                    totinlinelen := totinlinelen + 8;
                    totoutoflinelen := totoutoflinelen + tmp;
                END IF;

            WHEN oraTypeFloat THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                ftmp := tabData.columndata(colind).oraprecision;
                ftmp := (ftmp / ttFloatPrecisionCvt);
                tmp := TRUNC(ftmp) + 1;
                tabData.columndata(colind).ttdatatype := ttTypeFloat;
                scale := 0;
                tabData.columndata(colind).ttprecision := 
                    tabData.columndata(colind).oraprecision;
                tabData.columndata(colind).ttscale := scale;
                tabData.columndata(colind).ttdatalen := 
                    calcTTNumberStorage(tmp, scale);
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;

            WHEN oraTypeDate THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeDate;
                tabData.columndata(colind).ttdatalen := 7;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeBinaryFloat THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeBinaryFloat;
                tabData.columndata(colind).ttdatalen := 4;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeBinaryDouble THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeBinaryDouble;
                tabData.columndata(colind).ttdatalen := 8;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeTimestamp THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeTimestamp;
                tabData.columndata(colind).ttdatalen := 12;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeTimestampTZ THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeTimestamp;
                tabData.columndata(colind).ttdatalen := 12;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeTimestampLTZ THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeTimestamp;
                tabData.columndata(colind).ttdatalen := 12;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeChar THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeChar;
                tabData.columndata(colind).ttdatalen := 
                                tabData.columndata(colind).oradatalen;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;
               
            WHEN oraTypeNchar THEN
                totoralen := totoralen + tabData.columndata(colind).oradatalen;
                tabData.columndata(colind).ttdatatype := ttTypeNchar;
                tabData.columndata(colind).ttdatalen := 
                                tabData.columndata(colind).oradatalen;
                totinlinelen := totinlinelen + 
                                tabData.columndata(colind).ttdatalen;

            WHEN oraTypeClob THEN
                IF LOBMapping = LOBMappingNo THEN
                    tabData.columndata(colind).ttdatatype := ttTypeClob;
                ELSE
                    tabData.columndata(colind).ttdatatype := ttTypeVarchar2;
                    tabData.columndata(colind).oradatalen := ttVarMaxlen;
                    tabData.columndata(colind).defaschar := TRUE;
                    tabData.columndata(colind).ttcharlength := 
                        ttVarMaxlen / OraBytesPerChar;
                    tabData.columndata(colind).oracharlength :=
                        ttVarMaxlen / OraBytesPerChar;
                    totoralen := totoralen + 
                                 tabData.columndata(colind).oradatalen;
                    IF  tabData.columndata(colind).oradatalen > InlineLimit THEN
                        tmp := tabData.columndata(colind).oradatalen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        totinlinelen := totinlinelen + 8;
                        totoutoflinelen := totoutoflinelen + tmp;
                    ELSIF (totinlinelen + 
                           tabData.columndata(colind).oradatalen + 8)
                               < (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := 
                            tabData.columndata(colind).oradatalen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        totinlinelen := totinlinelen + 
                                        tabData.columndata(colind).ttdatalen;
                    ELSE
                        tmp := tabData.columndata(colind).oradatalen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        totinlinelen := totinlinelen + 8;
                        totoutoflinelen := totoutoflinelen + tmp;
                    END IF;
                END IF;
               
            WHEN oraTypeNclob THEN
                IF LOBMapping = LOBMappingNo THEN
                    tabData.columndata(colind).ttdatatype := ttTypeNclob;
                ELSE
                    tabData.columndata(colind).ttdatatype := ttTypeNvarchar2;
                    tabData.columndata(colind).oradatalen := ttVarMaxlen;
                    tabData.columndata(colind).defaschar := TRUE;
                    tabData.columndata(colind).ttcharlength := 
                        ttVarMaxlen / OraBytesPerNchar;
                    IF tabData.columndata(colind).ttcharlength > 
                           ttNvarMaxlen THEN
                        tabData.columndata(colind).ttcharlength := ttNvarMaxlen;
                        tabData.columndata(colind).oradatalen :=
                            tabData.columndata(colind).ttcharlength * 
                                OraBytesPerNchar;
                    END IF;
                    tabData.columndata(colind).oracharlength := 
                        tabData.columndata(colind).ttcharlength;
                    totoralen := totoralen + 
                                 tabData.columndata(colind).oradatalen;
                    IF  tabData.columndata(colind).oradatalen > InlineLimit THEN
                        tmp := tabData.columndata(colind).oradatalen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        totinlinelen := totinlinelen + 8;
                        totoutoflinelen := totoutoflinelen + tmp;
                    ELSIF (totinlinelen + 
                           tabData.columndata(colind).oradatalen + 8)
                               < (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := 
                            tabData.columndata(colind).oradatalen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        totinlinelen := totinlinelen + 
                                        tabData.columndata(colind).ttdatalen;
                    ELSE
                        tmp := tabData.columndata(colind).oradatalen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        totinlinelen := totinlinelen + 8;
                        totoutoflinelen := totoutoflinelen + tmp;
                    END IF;
                END IF;

            WHEN oraTypeBlob THEN
                IF LOBMapping = LOBMappingNo THEN
                    tabData.columndata(colind).ttdatatype := ttTypeBlob;
                ELSE
                    tabData.columndata(colind).ttdatatype := ttTypeVarbinary;
                    tabData.columndata(colind).defaschar := FALSE;
                    tabData.columndata(colind).oradatalen := ttVarMaxlen;
                    tabData.columndata(colind).ttcharlength :=  ttVarMaxlen;
                    tabData.columndata(colind).oracharlength := ttVarMaxlen;
                    totoralen := totoralen + 
                                 tabData.columndata(colind).oradatalen;
                    IF  tabData.columndata(colind).oradatalen > InlineLimit THEN
                        tmp := tabData.columndata(colind).oradatalen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        totinlinelen := totinlinelen + 8;
                        totoutoflinelen := totoutoflinelen + tmp;
                    ELSIF (totinlinelen + 
                           tabData.columndata(colind).oradatalen + 8)
                               < (ttMaxInlineLimit - 10) THEN
                        tabData.columndata(colind).ttdatalen := 
                            tabData.columndata(colind).oradatalen + 8;
                        tabData.columndata(colind).isoutofline := FALSE;
                        totinlinelen := totinlinelen + 
                                        tabData.columndata(colind).ttdatalen;
                    ELSE
                        tmp := tabData.columndata(colind).oradatalen + 16;
                        IF tmp < 32 THEN
                            tmp := 32;
                        END IF;
                        tabData.columndata(colind).ttdatalen := tmp + 8;
                        tabData.columndata(colind).isoutofline := TRUE;
                        totinlinelen := totinlinelen + 8;
                        totoutoflinelen := totoutoflinelen + tmp;
                    END IF;
                END IF;

        END CASE;

        END IF; -- skip

    END LOOP;

    tabData.totalorabytes := totoralen;
    tabData.totalinlinebytes := totinlinelen;
    tabData.totaloutoflinebytes := totoutoflinelen;

    retVal := resultSuccess;

    RETURN retVal;
END assignTimesTenTypes;

/*
 * buildReturnData
 *
 *     Assembles all the table and column data to be RETURNed.
 */

FUNCTION buildReturnData(
                         tabData IN tabledef,
                         errMsg OUT VARCHAR2
                        )
    RETURN PLS_INTEGER
IS
    retVal PLS_INTEGER := resultSuccess;
    maxprecision PLS_INTEGER;
    maxscale PLS_INTEGER;
    currcol columndef;
BEGIN
    -- encode table level data into gTableInfo
    gTableInfo := '';

    IF tabData.rowcount IS NOT NULL THEN
        gTableInfo := gTableInfo || tabData.rowcount;
    END IF;
    gTableInfo := gTableInfo || ',';

    IF tabData.iscompressed THEN
        gTableInfo := gTableInfo || qt || 'Y' || qt;
    ELSE
        gTableInfo := gTableInfo || qt || 'N' || qt;
    END IF;
    gTableInfo := gTableInfo || ',';

    IF tabData.totalorabytes IS NOT NULL THEN
        gTableInfo := gTableInfo || tabData.totalorabytes;
    END IF;
    gTableInfo := gTableInfo || ',';

    IF tabData.totalinlinebytes IS NOT NULL THEN
        gTableInfo := gTableInfo || tabData.totalinlinebytes;
    END IF;
    gTableInfo := gTableInfo || ',';

    IF tabData.totaloutoflinebytes IS NOT NULL THEN
        gTableInfo := gTableInfo || tabData.totaloutoflinebytes;
    END IF;
    gTableInfo := gTableInfo || ',';

    IF tabData.esttotalbytes IS NOT NULL THEN
        gTableInfo := gTableInfo || tabData.esttotalbytes;
    END IF;
    gTableInfo := gTableInfo || ',';

    IF tabData.esttotalbytescompressed IS NOT NULL THEN
        gTableInfo := gTableInfo || tabData.esttotalbytescompressed;
    END IF;

    -- encode column data into gColumnData
    FOR colind IN 1..tabData.numcolumns LOOP
        currcol := tabData.columndata(colind);

        -- compute maxprecision and maxscale
        IF currcol.maxfractionallen IS NULL THEN
            maxscale := 0;
        ELSE
            maxscale := currcol.maxfractionallen;
        END IF;

        IF currcol.maxintegrallen IS NULL THEN
            maxprecision := maxscale;
        ELSE
            maxprecision := currcol.maxintegrallen + maxscale;
        END IF;

        IF maxscale = 0 THEN
            maxscale := NULL;
        END IF;

        IF maxprecision = 0 THEN
            maxprecision := NULL;
        END IF;

        -- determine oratypetext
        currcol.oratypetext := getOraTypeText( currcol );

        -- determine tttypetext
        currcol.tttypetext := getTTTypeText( currcol );

        gColumnCount := gColumnCount + 1;
        gColumnData(gColumnCount) := '';

        IF currcol.name IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                qt || currcol.name || qt;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.skip THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                                         qt || 'Y' || qt;
        ELSE
            gColumnData(gColumnCount) := gColumnData(gColumnCount) ||
                                         qt || 'N' || qt;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.nullable THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) ||
                                         qt || 'Y' || qt;
        ELSE
            gColumnData(gColumnCount) := gColumnData(gColumnCount) ||
                                         qt || 'N' || qt;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.defaschar IS NOT NULL THEN
            IF currcol.defaschar THEN
                gColumnData(gColumnCount) := gColumnData(gColumnCount) ||
                                             qt || 'C' || qt;
            ELSE
                gColumnData(gColumnCount) := gColumnData(gColumnCount) ||
                                             qt || 'B' || qt;
            END IF;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.oradatatype IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.oradatatype);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.oratypetext IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                qt || currcol.oratypetext || qt;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.skip THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                                         ',,,,,,,,,,,,,,,,';
            CONTINUE;
        END IF;

        IF currcol.oraprecision IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.oraprecision);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.orascale IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.orascale);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF maxprecision IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(maxprecision);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF maxscale IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(maxscale);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.hasfractionalvalues IS NOT NULL THEN
            IF currcol.hasfractionalvalues THEN
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                                             qt || 'Y' || qt;
            ELSE
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                                             qt || 'N' || qt;
            END IF;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.oradatalen IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.oradatalen);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';
    
        IF (currcol.ttdatatype = ttTypeChar) OR
           (currcol.ttdatatype = ttTypeNchar) OR
           (currcol.ttdatatype = ttTypeVarchar2) OR
           (currcol.ttdatatype = ttTypeNvarchar2) THEN

            IF currcol.oracharlength IS NOT NULL THEN
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                    TO_CHAR(currcol.oracharlength);
            END IF;
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

            IF currcol.maxintegrallen IS NOT NULL THEN
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                    TO_CHAR(currcol.maxintegrallen);
            END IF;
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        ELSE
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',,';
        END IF;

        IF currcol.ttdatatype IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.ttdatatype);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.tttypetext IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                qt || currcol.tttypetext || qt;
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.ttprecision IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.ttprecision);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.ttscale IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.ttscale);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.ttdatalen IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.ttdatalen);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF (currcol.ttdatatype = ttTypeChar) OR
           (currcol.ttdatatype = ttTypeNchar) OR
           (currcol.ttdatatype = ttTypeVarchar2) OR
           (currcol.ttdatatype = ttTypeVarbinary) OR
           (currcol.ttdatatype = ttTypeNvarchar2) THEN

            IF currcol.ttcharlength IS NOT NULL THEN
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                    TO_CHAR(currcol.ttcharlength);
            END IF;

        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
            TO_CHAR(currcol.ttptrsize);
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';
        
        IF currcol.count IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.count);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.countdistinct IS NOT NULL THEN
            gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                TO_CHAR(currcol.countdistinct);
        END IF;
        gColumnData(gColumnCount) := gColumnData(gColumnCount) || ',';

        IF currcol.isoutofline IS NOT NULL THEN
            IF currcol.isoutofline THEN
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                                             qt || 'Y' || qt;
            ELSE
                gColumnData(gColumnCount) := gColumnData(gColumnCount) || 
                                             qt || 'N' || qt;
            END IF;
        END IF;

    END LOOP;

    <<fini>>
    RETURN retVal;
END buildReturnData;

/*
 * analyseTable
 *
 *     This is the main driver routine.
 */

PROCEDURE ANALYZE_TABLE(
                        OwnerName IN VARCHAR2,
                        TableName IN VARCHAR2,
                        AnalysisOptions IN VARCHAR2,
                        AnalysisResult OUT NUMBER,
                        AnalysisStatus OUT VARCHAR2,
                        TableInfo OUT VARCHAR2,
                        WarningCount OUT NUMBER,
                        Warnings OUT msgtable,
                        ColumnCount OUT NUMBER,
                        ColumnData OUT datatable
                       )
IS
    retVal PLS_INTEGER;
    errMsg VARCHAR2(512);
BEGIN
    -- initialise all RETURN values
    AnalysisResult := resultSuccess;
    AnalysisStatus := NULL;
    TableInfo := NULL;
    gTableInfo := NULL;
    WarningCount := 0;
    gWarningCount := 0;
    ColumnCount := 0;
    gColumnCount := 0;
    gWarnings.Delete();
    gColumnData.Delete();
    AnalysisQuery.results.Delete();

    -- check inputs
    IF (OwnerName IS NULL) OR (OwnerName = '') OR 
       (LENGTH(OwnerName) > maxSQLNameLen) THEN
        AnalysisResult := resultError;
        AnalysisStatus := 'Error: Internal: Invalid owner name ''' || OwnerName || '''';
        GOTO fini;
    END IF;

    IF (TableName IS NULL) OR (TableName = '') OR 
       (LENGTH(TableName) > maxSQLNameLen) THEN
        AnalysisResult := resultError;
        AnalysisStatus := 'Error: Internal: Invalid table name ''' || TableName || '''';
        GOTO fini;
    END IF;

    -- parse the AnalysisOptions string
    retVal := parseOptions( AnalysisOptions, errMsg );
    IF retVal != resultSuccess THEN
        AnalysisResult := resultError;
        AnalysisStatus := errMsg;
        GOTO fini;
    END IF;

    -- Determine DB bytes per char
    retVal := checkDBCharacterSet(errMsg);
    IF retVal != resultSuccess THEN
        AnalysisResult := resultError;
        AnalysisStatus := errMsg;
        GOTO fini;
    END IF;

    TableData.owner := OwnerName;
    TableData.name := TableName;

    -- get table metadata
    retVal := getColumnInfo( TableData, errMsg );
    IF retVal != resultSuccess THEN
        AnalysisResult := retVal;
        AnalysisStatus := errMsg;
        GOTO fini;
    END IF;

    -- do basic type mapping
    retVal := assignTimesTenTypes( TableData, errMsg );
    IF retVal != resultSuccess THEN
        AnalysisResult := retVal;
        AnalysisStatus := errMsg;
        GOTO fini;
    END IF;

    -- if agressive mapping or compression requested
    IF (NumMapping > typeMapNumNone) OR (VarMapping > typeMapVarNone) OR 
       (Compression > compressionNone) THEN

        retVal := analyseTableData( TableData, errMsg );
        IF retVal != resultSuccess THEN
            AnalysisResult := retVal;
            AnalysisStatus := errMsg;
            GOTO fini;
        END IF;

        IF (NumMapping > typeMapNumNone) OR (VarMapping > typeMapVarNone) THEN
            retVal := optimiseTimesTenTypes( TableData, errMsg );
            IF retVal != resultSuccess THEN
                AnalysisResult := retVal;
                AnalysisStatus := errMsg;
                GOTO fini;
            END IF;
        END IF;

        IF (Compression > compressionNone) THEN
            retVal := evaluateCompression( TableData, errMsg );
            IF retVal != resultSuccess THEN
                AnalysisResult := retVal;
                AnalysisStatus := errMsg;
                GOTO fini;
            END IF;
        ELSE
            retVal := computeTimesTenTableSize( TableData, errMsg );
            IF retVal != resultSuccess THEN
                AnalysisResult := retVal;
                AnalysisStatus := errMsg;
                GOTO fini;
            END IF;
        END IF;
    ELSE

        getRowCount( TableData );

        retVal := computeTimesTenTableSize( TableData, errMsg );
        IF retVal != resultSuccess THEN
            AnalysisResult := retVal;
            AnalysisStatus := errMsg;
            GOTO fini;
        END IF;
    END IF;

    retVal := buildReturnData( TableData, errMsg );
    IF retVal != resultSuccess THEN
        AnalysisResult := retVal;
        AnalysisStatus := errMsg;
        GOTO fini;
    END IF;

    <<fini>>
    -- set return values
    WarningCount := gWarningCount;
    Warnings := gWarnings;
    TableInfo := gTableInfo;
    ColumnCount := gColumnCount;
    ColumnData := gColumnData;
END ANALYZE_TABLE;

BEGIN
    ANALYZE_TABLE( 
                  -- Input parameters
                  :P_OWNER_NAME,
                  :P_TABLE_NAME,
                  :P_ANALYSIS_OPTIONS,

                  -- Returned results
                  :R_ANALYSIS_RESULT,
                  :R_ANALYSIS_STATUS,
                  :R_TABLE_INFO,
                  :R_WARNING_COUNT,
                  :R_WARNINGS,
                  :R_COLUMN_COUNT,
                  :R_COLUMN_DATA
                 );
END;
