<?xml version="1.0"?>
<!--

 Copyright (c) 2006, 2012, Oracle and/or its affiliates. All rights reserved. 

NAME
    kuacnstr.xsl
DESCRIPTION
    Convert constraint diffs in sxml TABLE diff document to ALTER_XML document

MODIFIED        MM/DD/YY
    abodge      02/16/12 - Fix ignore schema on FK constraints
    abodge      09/21/09 - CONSOLIDATE STYLESHEETS: MAKE_DIFF_READY -> CM_MODE
    rapayne     03/10/09 - bug 8227850: process all drops for all constraint
                           type to avoid rename constraint conflicts.
    rapayne     01/12/09 - bug 7595899: pick up some changes for CM
    rapayne     09/17/08 - bug 7374528: Use RELY when 'adding' a constraint.
    lbarton     07/03/08 - bug 5709159: drop/modify unnamed constraint
    rapayne     02/29/08 - Add support for AlterConstraints for OBJECT_TABLEs.
                         - fix COL_LIST//@src syntax
                         - force drop/create for changes to INITIALLY_DEFERRED and
                           changes to unnamed constraints.
                         - merge CM changes.
                         - use common SourceName.
                         - add support for RELY diffs in modify path.
    rapayne     02/27/08 - Generate drop/create for diffs in:
                                 DEFERRABLE, INITIALLY DEFERRABLE
    lbarton     11/28/07 - bug 6474004: RELY
    rapayne     10/19/07 - Do not drop-create diffs in novalidate.
    rapayne     01/19/07 - Only call ModifyConstraints if there have been changes.
    rapayne     12/21/06 - move AlterCOL_LIST to kuacomm.xsl
    lbarton     12/18/06 - drop unnamed constraints
    htseng      11/21/06 - more work for using index
    lbarton     11/20/06 - remove ENABLE VALIDATE
    htseng      11/16/06 - add index for constraint_state 
    lbarton     10/17/06 - still more
    htseng      10/10/06 - more work
    htseng      10/03/06 - more work
    htseng      09/26/06 - constraints
    lbarton     09/22/05 - Initial version
 -->
<xsl:stylesheet version="1.0" xmlns:sxml="http://xmlns.oracle.com/ku" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
 <!-- Top level imports -->
 <xsl:import href="kuacomm.xsl"/>
 <xsl:import href="kuatprop.xsl"/>
 <!-- Templates -->
 <xsl:template name="ProcessDropConstraints">
  <xsl:param name="TableNode" select="''"/>
  <xsl:param name="ParentNode" select="''"/>
  <!-- *******************************************************************
Template: ProcessDropConstraints
Description: cycle through all of the constraint types and process any 
 approprate drops. This will avoid any ADDs with same name constraints
 from producing runtime errors.
Parameters:
 TableNode - Parent node of SCHEMA, NAME, constraint lists
 ParentNode - Parent of *CONSTRAINT_LIST
                       RELATIONAL_TABLE || OBJECT_TABLE
******************************************************************** -->
  <!-- PRIMARY_KEY_CONSTRAINT_LIST -->
  <xsl:for-each select="$ParentNode/sxml:PRIMARY_KEY_CONSTRAINT_LIST/sxml:PRIMARY_KEY_CONSTRAINT_LIST_ITEM">
   <xsl:if test="./@src='1' or ../@src='1'">
    <xsl:call-template name="DropConstraints">
     <xsl:with-param name="ParentNode" select="$TableNode"/>
     <xsl:with-param name="ListItem" select="."/>
     <xsl:with-param name="ConsType" select="'PRIMARY'"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
  <!-- FOREIGN_KEY_CONSTRAINT_LIST -->
  <xsl:for-each select="$ParentNode/sxml:FOREIGN_KEY_CONSTRAINT_LIST/sxml:FOREIGN_KEY_CONSTRAINT_LIST_ITEM">
   <xsl:if test="./@src='1' or ../@src='1'">
    <xsl:call-template name="DropConstraints">
     <xsl:with-param name="ParentNode" select="$TableNode"/>
     <xsl:with-param name="ListItem" select="."/>
     <xsl:with-param name="ConsType" select="'FOREIGN'"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
  <!-- UNIQUE_KEY_CONSTRAINT_LIST -->
  <xsl:for-each select="$ParentNode/sxml:UNIQUE_KEY_CONSTRAINT_LIST/sxml:UNIQUE_KEY_CONSTRAINT_LIST_ITEM">
   <xsl:if test="./@src='1'  or ../@src='1'">
    <xsl:call-template name="DropConstraints">
     <xsl:with-param name="ParentNode" select="$TableNode"/>
     <xsl:with-param name="ListItem" select="."/>
     <xsl:with-param name="ConsType" select="'UNIQUE'"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
  <!-- CHECK_CONSTRAINT_LIST -->
  <xsl:for-each select="$ParentNode/sxml:CHECK_CONSTRAINT_LIST/sxml:CHECK_CONSTRAINT_LIST_ITEM">
   <xsl:if test="./@src='1' or ../@src='1'">
    <xsl:call-template name="DropConstraints">
     <xsl:with-param name="ParentNode" select="$TableNode"/>
     <xsl:with-param name="ListItem" select="."/>
     <xsl:with-param name="ConsType" select="'CHECK'"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
  <!-- SCOPE_CONSTRAINT_LIST -->
  <xsl:for-each select="$ParentNode/sxml:SCOPE_CONSTRAINT_LIST/sxml:SCOPE_CONSTRAINT_LIST_ITEM">
   <xsl:if test="./@src='1' or ../@src='1'">
    <xsl:call-template name="DropConstraints">
     <xsl:with-param name="ParentNode" select="$TableNode"/>
     <xsl:with-param name="ListItem" select="."/>
     <xsl:with-param name="ConsType" select="'SCOPE'"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
  <!-- ROWID_CONSTRAINT_LIST -->
 </xsl:template>
 <xsl:template name="AlterConstraints">
  <xsl:param name="TableNode" select="''"/>
  <xsl:param name="ParentNode" select="''"/>
  <!-- *******************************************************************
Template: AlterConstraints
Parameters:
 TableNode - Parent node of SCHEMA, NAME, constraint lists
 ParentNode - Parent of *CONSTRAINT_LIST
                       RELATIONAL_TABLE || OBJECT_TABLE
******************************************************************** -->
  <!-- It is necessary to drop designated constraints (i.e., src=1) for each type
   of constraint to avoid renamed constraint conflicts
-->
  <xsl:call-template name="ProcessDropConstraints">
   <xsl:with-param name="ParentNode" select="$ParentNode"/>
   <xsl:with-param name="TableNode" select="$TableNode"/>
  </xsl:call-template>
  <!-- PRIMARY_KEY_CONSTRAINT_LIST -->
  <xsl:if test="$ParentNode/sxml:PRIMARY_KEY_CONSTRAINT_LIST">
   <xsl:call-template name="DoAlterConstraints">
    <xsl:with-param name="ParentNode" select="$TableNode"/>
    <xsl:with-param name="TableTypeNode" select="$ParentNode"/>
    <xsl:with-param name="ListItemNodes" select="$ParentNode/sxml:PRIMARY_KEY_CONSTRAINT_LIST/sxml:PRIMARY_KEY_CONSTRAINT_LIST_ITEM"/>
    <xsl:with-param name="ConsType" select="'PRIMARY'"/>
    <xsl:with-param name="DoAll">
     <xsl:choose>
      <xsl:when test="$ParentNode/sxml:PRIMARY_KEY_CONSTRAINT_LIST/@src='2'">ADD</xsl:when>
      <xsl:otherwise>N</xsl:otherwise>
     </xsl:choose>
    </xsl:with-param>
   </xsl:call-template>
  </xsl:if>
  <!-- FOREIGN_KEY_CONSTRAINT_LIST -->
  <xsl:if test="$ParentNode/sxml:FOREIGN_KEY_CONSTRAINT_LIST">
   <xsl:call-template name="DoAlterConstraints">
    <xsl:with-param name="ParentNode" select="$TableNode"/>
    <xsl:with-param name="TableTypeNode" select="'$ParentNode'"/>
    <xsl:with-param name="ListItemNodes" select="$ParentNode/sxml:FOREIGN_KEY_CONSTRAINT_LIST/sxml:FOREIGN_KEY_CONSTRAINT_LIST_ITEM"/>
    <xsl:with-param name="ConsType" select="'FOREIGN'"/>
    <xsl:with-param name="DoAll">
     <xsl:choose>
      <xsl:when test="$ParentNode/sxml:FOREIGN_KEY_CONSTRAINT_LIST/@src='2'">ADD</xsl:when>
      <xsl:otherwise>N</xsl:otherwise>
     </xsl:choose>
    </xsl:with-param>
   </xsl:call-template>
  </xsl:if>
  <!-- UNIQUE_KEY_CONSTRAINT_LIST -->
  <xsl:if test="$ParentNode/sxml:UNIQUE_KEY_CONSTRAINT_LIST">
   <xsl:call-template name="DoAlterConstraints">
    <xsl:with-param name="ParentNode" select="$TableNode"/>
    <xsl:with-param name="TableTypeNode" select="'$ParentNode'"/>
    <xsl:with-param name="ListItemNodes" select="$ParentNode/sxml:UNIQUE_KEY_CONSTRAINT_LIST/sxml:UNIQUE_KEY_CONSTRAINT_LIST_ITEM"/>
    <xsl:with-param name="ConsType" select="'UNIQUE'"/>
    <xsl:with-param name="DoAll">
     <xsl:choose>
      <xsl:when test="$ParentNode//sxml:UNIQUE_KEY_CONSTRAINT_LIST/@src='2'">ADD</xsl:when>
      <xsl:otherwise>N</xsl:otherwise>
     </xsl:choose>
    </xsl:with-param>
   </xsl:call-template>
  </xsl:if>
  <!-- CHECK_CONSTRAINT_LIST -->
  <xsl:if test="$ParentNode/sxml:CHECK_CONSTRAINT_LIST">
   <xsl:call-template name="DoAlterConstraints">
    <xsl:with-param name="ParentNode" select="$TableNode"/>
    <xsl:with-param name="TableTypeNode" select="'$ParentNode'"/>
    <xsl:with-param name="ListItemNodes" select="$ParentNode/sxml:CHECK_CONSTRAINT_LIST/sxml:CHECK_CONSTRAINT_LIST_ITEM"/>
    <xsl:with-param name="ConsType" select="'CHECK'"/>
    <xsl:with-param name="DoAll">
     <xsl:choose>
      <xsl:when test="$ParentNode/sxml:CHECK_CONSTRAINT_LIST/@src='2'">ADD</xsl:when>
      <xsl:otherwise>N</xsl:otherwise>
     </xsl:choose>
    </xsl:with-param>
   </xsl:call-template>
  </xsl:if>
  <!-- SCOPE_CONSTRAINT_LIST -->
  <xsl:if test="$ParentNode/sxml:SCOPE_CONSTRAINT_LIST">
   <xsl:call-template name="DoAlterConstraints">
    <xsl:with-param name="ParentNode" select="$TableNode"/>
    <xsl:with-param name="TableTypeNode" select="'$ParentNode'"/>
    <xsl:with-param name="ListItemNodes" select="$ParentNode/sxml:SCOPE_CONSTRAINT_LIST/sxml:SCOPE_CONSTRAINT_LIST_ITEM"/>
    <xsl:with-param name="ConsType" select="'SCOPE'"/>
    <xsl:with-param name="DoAll">
     <xsl:choose>
      <xsl:when test="$ParentNode/sxml:SCOPE_CONSTRAINT_LIST/@src='2'">ADD</xsl:when>
      <xsl:otherwise>N</xsl:otherwise>
     </xsl:choose>
    </xsl:with-param>
   </xsl:call-template>
  </xsl:if>
  <!-- ROWID_CONSTRAINT_LIST -->
 </xsl:template>
 <xsl:template name="DoAlterConstraints">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="TableTypeNode" select="''"/>
  <xsl:param name="ListItemNodes" select="''"/>
  <xsl:param name="ConsType">N</xsl:param>
  <xsl:param name="DoAll">N</xsl:param>
  <!-- *******************************************************************
Template: DoAlterConstraints
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 TableTypeNode - <RELATIONAL_TABLE> | <OBJECT_TABLE>
 ListItemNodes - CONSTRAINT_LIST_ITEM nodes
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
 DoAll - "DROP" = entire list is in doc1, not in doc2
         "ADD"  = entire list is in doc2, not in doc1
         "N"    = otherwise
******************************************************************** -->
  <xsl:for-each select="$ListItemNodes">
   <xsl:choose>
    <!-- Drop
          note: *_ITEM/@src='1' are  processed in DropAllConstraints and should have already been dropped.
  -->
    <xsl:when test="$DoAll='DROP'">
     <xsl:call-template name="DropConstraints">
      <xsl:with-param name="ParentNode" select="$ParentNode"/>
      <xsl:with-param name="ListItem" select="."/>
      <xsl:with-param name="ConsType" select="$ConsType"/>
     </xsl:call-template>
    </xsl:when>
    <!-- Add -->
    <xsl:when test="$DoAll='ADD' or @src='2'">
     <xsl:call-template name="AddConstraints">
      <xsl:with-param name="ParentNode" select="$ParentNode"/>
      <xsl:with-param name="ListItem" select="."/>
      <xsl:with-param name="ConsType" select="$ConsType"/>
     </xsl:call-template>
    </xsl:when>
    <!-- Condition is different -->
    <!-- Modify or Rename -->
    <!-- look for descendant nodes with a value1 or src attribute -->
    <xsl:when test="(.//@value1 or .//@src) and not(./@src='1')">
     <xsl:call-template name="ModifyConstraints">
      <xsl:with-param name="ParentNode" select="$ParentNode"/>
      <xsl:with-param name="TableTypeNode" select="$TableTypeNode"/>
      <xsl:with-param name="ListItem" select="."/>
      <xsl:with-param name="ConsType" select="$ConsType"/>
     </xsl:call-template>
    </xsl:when>
   </xsl:choose>
  </xsl:for-each>
 </xsl:template>
 <xsl:template name="DropConstraints">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType">N</xsl:param>
  <!-- *******************************************************************
Template: DropConstraints - wrapper which calls DropConstraint for 
 Parse and SQL actions.
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
******************************************************************** -->
  <xsl:element name="ALTER_LIST_ITEM">
   <xsl:call-template name="DropConstraint">
    <xsl:with-param name="ParentNode" select="$ParentNode"/>
    <xsl:with-param name="ListItem" select="$ListItem"/>
    <xsl:with-param name="ConsType" select="$ConsType"/>
    <xsl:with-param name="Action">PARSE</xsl:with-param>
   </xsl:call-template>
   <xsl:call-template name="DropConstraint">
    <xsl:with-param name="ParentNode" select="$ParentNode"/>
    <xsl:with-param name="ListItem" select="$ListItem"/>
    <xsl:with-param name="ConsType" select="$ConsType"/>
    <xsl:with-param name="Action">SQL</xsl:with-param>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="DropConstraint">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: DropConstraint - process (i.e., drop) current constraint
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
 Action   - PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
					$PRS_CONSTRAINT_TYPE=1 or
					$PRS_CONSTRAINT_STATE=1 or
                                        $PRS_ALTERABLE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$ListItem"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1">
       <xsl:choose>
        <xsl:when test="sxml:NAME/@value1">
         <xsl:value-of select="sxml:NAME/@value1"/>
        </xsl:when>
        <xsl:when test="@NAME">
         <xsl:value-of select="@NAME"/>
        </xsl:when>
        <xsl:otherwise>
         <xsl:value-of select="sxml:NAME"/>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:with-param>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">DROP_CONSTRAINT</xsl:with-param>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CONSTRAINT_TYPE"/>
      <xsl:with-param name="Item">CONSTRAINT_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">
       <xsl:value-of select="$ConsType"/>
      </xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:call-template name="DropConstr">
      <xsl:with-param name="ParentNode" select="$ParentNode"/>
      <xsl:with-param name="ListItem" select="$ListItem"/>
      <xsl:with-param name="ConsType" select="$ConsType"/>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="DropConstr">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType" select="''"/>
  <!-- *******************************************************************
Template: DropConstr - generate the SQL_LIST_ITEM for the drop constraint
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of table SCHEMA, NAME
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
******************************************************************** -->
  <xsl:element name="SQL_LIST_ITEM">
   <xsl:choose>
    <xsl:when test="$ListItem/sxml:NAME">
     <!-- named constraint -->
     <xsl:element name="TEXT">
      <xsl:text>ALTER TABLE </xsl:text>
      <xsl:call-template name="SchemaName">
       <xsl:with-param name="ParentNode" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:text> DROP CONSTRAINT </xsl:text>
      <xsl:call-template name="SourceName">
       <xsl:with-param name="NameNode" select="$ListItem/sxml:NAME"/>
      </xsl:call-template>
     </xsl:element>
    </xsl:when>
    <xsl:when test="$ConsType='PRIMARY'">
     <!-- unnamed primary key constraint -->
     <xsl:element name="TEXT">
      <xsl:text>ALTER TABLE </xsl:text>
      <xsl:call-template name="SchemaName">
       <xsl:with-param name="ParentNode" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:text> DROP PRIMARY KEY</xsl:text>
     </xsl:element>
    </xsl:when>
    <xsl:when test="$ConsType='UNIQUE'">
     <!-- unnamed unique constraint -->
     <xsl:element name="TEXT">
      <xsl:text>ALTER TABLE </xsl:text>
      <xsl:call-template name="SchemaName">
       <xsl:with-param name="ParentNode" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:text> DROP </xsl:text>
      <xsl:call-template name="GetConstraint">
       <xsl:with-param name="ParentNode" select="$ListItem"/>
       <xsl:with-param name="ColListItems" select="$ListItem/sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
       <xsl:with-param name="ConsType" select="$ConsType"/>
      </xsl:call-template>
     </xsl:element>
    </xsl:when>
    <xsl:when test="$ListItem/@NAME">
     <!-- NAME attribute in constraint list item element;
         this was generated by CM_MODE transform param -->
     <xsl:element name="TEXT">
      <xsl:text>ALTER TABLE </xsl:text>
      <xsl:call-template name="SchemaName">
       <xsl:with-param name="ParentNode" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:text> DROP CONSTRAINT </xsl:text>
      <xsl:value-of select="$ListItem/@NAME"/>
     </xsl:element>
    </xsl:when>
    <xsl:when test="$ConsType='CHECK'">
     <!-- unnamed check constraint -->
     <xsl:element name="TEXT">
      <xsl:text>DECLARE
 cname VARCHAR2(30);
BEGIN
 cname := dbms_metadata.get_check_constraint_name('TABLE','</xsl:text>
      <xsl:call-template name="UnquotedSourceName">
       <xsl:with-param name="NameNode" select="$ParentNode/sxml:SCHEMA"/>
      </xsl:call-template>
      <xsl:text>','</xsl:text>
      <xsl:call-template name="UnquotedSourceName">
       <xsl:with-param name="NameNode" select="$ParentNode/sxml:NAME"/>
      </xsl:call-template>
      <xsl:text>','</xsl:text>
     </xsl:element>
     <xsl:choose>
      <xsl:when test="$ListItem/sxml:CONDITION">
       <xsl:element name="TEXT">
        <xsl:call-template name="EscapeString">
         <xsl:with-param name="TheString" select="$ListItem/sxml:CONDITION"/>
        </xsl:call-template>
        <xsl:text>',FALSE);</xsl:text>
       </xsl:element>
      </xsl:when>
      <xsl:when test="$ListItem/sxml:PARSED_CONDITION">
       <xsl:element name="ELEMENTS">
        <xsl:copy-of select="$ListItem/sxml:PARSED_CONDITION/*"/>
       </xsl:element>
       <xsl:element name="TEXT">
        <xsl:text>',TRUE);</xsl:text>
       </xsl:element>
      </xsl:when>
     </xsl:choose>
     <xsl:element name="TEXT">
      <xsl:text>
 execute immediate 'ALTER TABLE </xsl:text>
      <xsl:call-template name="SchemaName">
       <xsl:with-param name="ParentNode" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:text> DROP CONSTRAINT ' || cname;
END;</xsl:text>
     </xsl:element>
    </xsl:when>
    <xsl:when test="$ConsType='FOREIGN'">
     <!-- unnamed foreign key constraint -->
     <xsl:element name="TEXT">
      <xsl:text>DECLARE
 h     NUMBER;
 cname VARCHAR2(30);
BEGIN
 h := dbms_metadata.open_get_fk_constraint_name('TABLE','</xsl:text>
      <xsl:call-template name="UnquotedSourceName">
       <xsl:with-param name="NameNode" select="$ParentNode/sxml:SCHEMA"/>
      </xsl:call-template>
      <xsl:text>','</xsl:text>
      <xsl:call-template name="UnquotedSourceName">
       <xsl:with-param name="NameNode" select="$ParentNode/sxml:NAME"/>
      </xsl:call-template>
      <xsl:text>','</xsl:text>
      <xsl:value-of select="$ListItem/sxml:REFERENCES/sxml:SCHEMA"/>
      <xsl:text>','</xsl:text>
      <xsl:value-of select="$ListItem/sxml:REFERENCES/sxml:NAME"/>
      <xsl:text>');</xsl:text>
      <xsl:for-each select="$ListItem/sxml:COL_LIST/sxml:COL_LIST_ITEM">
       <xsl:text> 
 dbms_metadata.set_fk_constraint_col_pair(h,'</xsl:text>
       <xsl:value-of select="sxml:NAME"/>
       <xsl:text>','</xsl:text>
       <xsl:call-template name="EmitReferenceName">
        <xsl:with-param name="ColList" select="$ListItem/sxml:REFERENCES/sxml:COL_LIST"/>
        <xsl:with-param name="pos" select="position()"/>
       </xsl:call-template>
       <xsl:text>');</xsl:text>
      </xsl:for-each>
      <xsl:text> 
 cname := dbms_metadata.get_fk_constraint_name(h);
 execute immediate 'ALTER TABLE </xsl:text>
      <xsl:call-template name="SchemaName">
       <xsl:with-param name="ParentNode" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:text> DROP CONSTRAINT ' || cname;
END;</xsl:text>
     </xsl:element>
    </xsl:when>
    <xsl:otherwise>
     <xsl:element name="TEXT">
      <xsl:text>-- ERROR: Unable to drop constraint of type </xsl:text>
      <xsl:value-of select="$ConsType"/>
     </xsl:element>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:element>
 </xsl:template>
 <xsl:template name="EmitReferenceName">
  <xsl:param name="ColList" select="''"/>
  <xsl:param name="pos">0</xsl:param>
  <xsl:value-of select="$ColList/sxml:COL_LIST_ITEM[$pos]/sxml:NAME"/>
 </xsl:template>
 <xsl:template name="AddConstraints">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType">N</xsl:param>
  <!-- *******************************************************************
Template: AddConstraints
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
******************************************************************** -->
  <xsl:element name="ALTER_LIST_ITEM">
   <xsl:call-template name="AddConstraint">
    <xsl:with-param name="ParentNode" select="$ParentNode"/>
    <xsl:with-param name="ListItem" select="$ListItem"/>
    <xsl:with-param name="ConsType" select="$ConsType"/>
    <xsl:with-param name="Action">PARSE</xsl:with-param>
   </xsl:call-template>
   <xsl:call-template name="AddConstraint">
    <xsl:with-param name="ParentNode" select="$ParentNode"/>
    <xsl:with-param name="ListItem" select="$ListItem"/>
    <xsl:with-param name="ConsType" select="$ConsType"/>
    <xsl:with-param name="Action">SQL</xsl:with-param>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="AddConstraint">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AddConstraint
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
 Action   - PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
					$PRS_CONSTRAINT_TYPE=1 or
					$PRS_CONSTRAINT_STATE=1 or
                                        $PRS_ALTERABLE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$ListItem"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1">
       <xsl:choose>
        <xsl:when test="sxml:NAME/@value1">
         <xsl:value-of select="sxml:NAME/@value1"/>
        </xsl:when>
        <xsl:when test="@NAME">
         <xsl:value-of select="@NAME"/>
        </xsl:when>
        <xsl:otherwise>
         <xsl:value-of select="sxml:NAME"/>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:with-param>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">ADD_CONSTRAINT</xsl:with-param>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CONSTRAINT_TYPE"/>
      <xsl:with-param name="Item">CONSTRAINT_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">
       <xsl:value-of select="$ConsType"/>
      </xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER TABLE </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$ParentNode"/>
       </xsl:call-template>
       <xsl:text> ADD </xsl:text>
       <xsl:if test="sxml:NAME">
        <xsl:text>CONSTRAINT </xsl:text>
        <xsl:call-template name="QuotedName">
         <xsl:with-param name="NameNode" select="sxml:NAME"/>
        </xsl:call-template>
       </xsl:if>
       <xsl:call-template name="GetConstraint">
        <xsl:with-param name="ParentNode" select="$ListItem"/>
        <xsl:with-param name="ColListItems" select="sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
        <xsl:with-param name="ConsType" select="$ConsType"/>
       </xsl:call-template>
       <xsl:if test="$ConsType!='SCOPE'">
        <xsl:call-template name="AlterConstraintState">
         <xsl:with-param name="ParentNode" select="$ListItem"/>
         <xsl:with-param name="TypeNode">ADD</xsl:with-param>
        </xsl:call-template>
       </xsl:if>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="ModifyConstraints">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="TableTypeNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType">N</xsl:param>
  <!-- *******************************************************************
Template: ModifyConstraints
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 TableTypeNode - <RELATIONAL_TABLE> | <OBJECT_TABLE>
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
******************************************************************** -->
  <xsl:element name="ALTER_LIST_ITEM">
   <xsl:call-template name="ModifyConstraint">
    <xsl:with-param name="ParentNode" select="$ParentNode"/>
    <xsl:with-param name="TableTypeNode" select="$TableTypeNode"/>
    <xsl:with-param name="ListItem" select="$ListItem"/>
    <xsl:with-param name="ConsType" select="$ConsType"/>
    <xsl:with-param name="Action">PARSE</xsl:with-param>
   </xsl:call-template>
   <xsl:call-template name="ModifyConstraint">
    <xsl:with-param name="ParentNode" select="$ParentNode"/>
    <xsl:with-param name="TableTypeNode" select="$TableTypeNode"/>
    <xsl:with-param name="ListItem" select="$ListItem"/>
    <xsl:with-param name="ConsType" select="$ConsType"/>
    <xsl:with-param name="Action">SQL</xsl:with-param>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="ModifyConstraint">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="TableTypeNode" select="''"/>
  <xsl:param name="ListItem" select="''"/>
  <xsl:param name="ConsType" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: ModifyConstraint
Current node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 TableTypeNode - <RELATIONAL_TABLE> | <OBJECT_TABLE>
 ListItem - CONSTRAINT_LIST_ITEM node
 ConsType - 'PRIMARY', 'FOREIGN', 'UNIQUE', 'CHECK', 'SCOPE'
 Action   - PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
					$PRS_CONSTRAINT_TYPE=1 or
					$PRS_CONSTRAINT_STATE=1 or
                                        $PRS_ALTERABLE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$ListItem"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1">
       <xsl:choose>
        <xsl:when test="$CM_MODE=1">
         <xsl:choose>
          <!-- NOTE - different selection order -->
          <xsl:when test="sxml:NAME">
           <xsl:value-of select="sxml:NAME"/>
          </xsl:when>
          <xsl:when test="@NAME">
           <xsl:value-of select="@NAME"/>
          </xsl:when>
          <xsl:otherwise>
           <xsl:value-of select="sxml:NAME/@value1"/>
          </xsl:otherwise>
         </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
         <xsl:choose>
        <xsl:when test="sxml:NAME/@value1">
         <xsl:value-of select="sxml:NAME/@value1"/>
        </xsl:when>
        <xsl:otherwise>
         <xsl:value-of select="sxml:NAME"/>
        </xsl:otherwise>
       </xsl:choose>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:with-param>
     </xsl:call-template>
     <xsl:choose>
      <xsl:when test="sxml:COL_LIST/sxml:COL_LIST_ITEM/sxml:NAME/@value1
                     or sxml:CONDITION/@value1 
                     or $ListItem/sxml:COL_LIST//@src
                     or sxml:REFERENCES/sxml:NAME/@value1
                     or sxml:DEFERRABLE/@src
                     or sxml:INITIALLY_DEFERRED/@src">
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">DROP_CREATE_CONSTRAINT</xsl:with-param>
       </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
       <xsl:variable name="ClauseType">
        <xsl:choose>
         <xsl:when test="sxml:DISABLE/@src='2'">DISABLE_CONSTRAINT</xsl:when>
         <xsl:when test="sxml:DISABLE/@src='1'">ENABLE_CONSTRAINT</xsl:when>
         <xsl:otherwise>MODIFY_CONSTRAINT</xsl:otherwise>
        </xsl:choose>
       </xsl:variable>
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1" select="$ClauseType"/>
       </xsl:call-template>
      </xsl:otherwise>
     </xsl:choose>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CONSTRAINT_TYPE"/>
      <xsl:with-param name="Item">CONSTRAINT_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">
       <xsl:value-of select="$ConsType"/>
      </xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:choose>
      <!-- look for descendant nodes with a value1 or src attribute -->
      <xsl:when test="sxml:COL_LIST/sxml:COL_LIST_ITEM/sxml:NAME/@value1
                     or sxml:CONDITION/@value1 
                     or sxml:COL_LIST/sxml:COL_LIST_ITEM//@src
                     or sxml:REFERENCES/sxml:NAME/@value1
                     or sxml:DEFERRABLE/@src
                     or sxml:INITIALLY_DEFERRED/@src">
       <xsl:call-template name="DropConstr">
        <xsl:with-param name="ParentNode" select="$ParentNode"/>
        <xsl:with-param name="ListItem" select="$ListItem"/>
        <xsl:with-param name="ConsType" select="$ConsType"/>
       </xsl:call-template>
       <xsl:element name="SQL_LIST_ITEM">
        <xsl:element name="TEXT">
         <xsl:text>ALTER TABLE </xsl:text>
         <xsl:call-template name="SchemaName">
          <xsl:with-param name="ParentNode" select="$ParentNode"/>
         </xsl:call-template>
         <xsl:text> ADD </xsl:text>
         <xsl:if test="sxml:NAME">
          <xsl:text>CONSTRAINT </xsl:text>
          <xsl:call-template name="QuotedName">
           <xsl:with-param name="NameNode" select="sxml:NAME"/>
          </xsl:call-template>
         </xsl:if>
         <xsl:call-template name="GetConstraint">
          <xsl:with-param name="ParentNode" select="$ListItem"/>
          <xsl:with-param name="ColListItems" select="sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
          <xsl:with-param name="ConsType" select="$ConsType"/>
         </xsl:call-template>
         <!-- DEFERRABLE need do in create/add statement -->
         <xsl:if test="$ListItem/sxml:DEFERRABLE">
          <xsl:choose>
           <xsl:when test="$ListItem/sxml:DEFERRABLE/@src='1'"/>
           <xsl:otherwise>
            <xsl:text> DEFERRABLE</xsl:text>
           </xsl:otherwise>
          </xsl:choose>
         </xsl:if>
         <xsl:call-template name="AlterConstraintState">
          <xsl:with-param name="ParentNode" select="$ListItem"/>
          <xsl:with-param name="TypeNode">ADD</xsl:with-param>
         </xsl:call-template>
        </xsl:element>
       </xsl:element>
      </xsl:when>
      <xsl:when test="$ListItem/sxml:INITIALLY_DEFERRED/@src='2' or
              $ListItem/sxml:DISABLE/@src or
              $ListItem/sxml:NOVALIDATE/@src or
              ($ListItem/sxml:USING_INDEX/@src='2' and not($TableTypeNode/sxml:PHYSICAL_PROPERTIES/sxml:INDEX_ORGANIZED_TABLE/@src)) or
       $ListItem/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:PCTFREE/@value1 or
       $ListItem/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:INITRANS/@value1 or
       $ListItem/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:STORAGE/sxml:NEXT/@value1 or
       $ListItem/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:STORAGE/sxml:MINEXTENTS/@value1 or
       $ListItem/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING/@src or
       $ListItem/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM//@value1 or
       $ListItem/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM/sxml:SEGMENT_ATTRIBUTES//@value1 or
       $ListItem/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING/@src or
       $ListItem/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM//@value1 or
       $ListItem/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM/sxml:SEGMENT_ATTRIBUTES//@value1">
       <xsl:element name="SQL_LIST_ITEM">
        <xsl:element name="TEXT">
         <xsl:text>ALTER TABLE </xsl:text>
         <xsl:call-template name="SchemaName">
          <xsl:with-param name="ParentNode" select="$ParentNode"/>
         </xsl:call-template>
         <!-- Need to handle named and unnamed constraints differently -->
         <xsl:choose>
          <xsl:when test="sxml:NAME and not (sxml:NAME/@src='2')">
           <!-- Named constraint -->
           <xsl:text> MODIFY CONSTRAINT </xsl:text>
           <xsl:call-template name="SourceName">
            <xsl:with-param name="NameNode" select="sxml:NAME"/>
           </xsl:call-template>
          </xsl:when>
          <xsl:otherwise>
           <!-- Unnamed constraint -->
           <xsl:text> MODIFY</xsl:text>
           <xsl:choose>
            <xsl:when test="$ConsType='PRIMARY' or $ConsType='UNIQUE'">
             <xsl:call-template name="GetConstraint">
              <xsl:with-param name="ParentNode" select="$ListItem"/>
              <xsl:with-param name="ColListItems" select="$ListItem/sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
              <xsl:with-param name="ConsType" select="$ConsType"/>
              <xsl:with-param name="ForModify" select="TRUE"/>
             </xsl:call-template>
            </xsl:when>
            <xsl:otherwise>
             <xsl:text> CANNOT MODIFY UNNAMED CONSTRAINT OF TYPE </xsl:text>
             <xsl:value-of select="$ConsType"/>
            </xsl:otherwise>
           </xsl:choose>
          </xsl:otherwise>
         </xsl:choose>
         <xsl:call-template name="AlterConstraintState">
          <xsl:with-param name="ParentNode" select="$ListItem"/>
         </xsl:call-template>
        </xsl:element>
       </xsl:element>
       <!-- first rename it then modify -->
       <xsl:if test="sxml:NAME/@value1">
        <xsl:element name="SQL_LIST_ITEM">
         <xsl:element name="TEXT">
          <xsl:text>ALTER TABLE </xsl:text>
          <xsl:call-template name="SchemaName">
           <xsl:with-param name="ParentNode" select="$ParentNode"/>
          </xsl:call-template>
          <xsl:text> RENAME CONSTRAINT </xsl:text>
          <xsl:call-template name="QuotedName">
           <xsl:with-param name="NameNode" select="sxml:NAME/@value1"/>
          </xsl:call-template>
          <xsl:text> TO </xsl:text>
          <xsl:call-template name="QuotedName">
           <xsl:with-param name="NameNode" select="sxml:NAME"/>
          </xsl:call-template>
         </xsl:element>
        </xsl:element>
       </xsl:if>
      </xsl:when>
      <xsl:otherwise>
       <!-- Rename: nothing diff except constraint name -->
       <xsl:if test="sxml:NAME/@value1">
        <xsl:element name="SQL_LIST_ITEM">
         <xsl:element name="TEXT">
          <xsl:text>ALTER TABLE </xsl:text>
          <xsl:call-template name="SchemaName">
           <xsl:with-param name="ParentNode" select="$ParentNode"/>
          </xsl:call-template>
          <xsl:text> RENAME CONSTRAINT </xsl:text>
          <xsl:call-template name="QuotedName">
           <xsl:with-param name="NameNode" select="sxml:NAME/@value1"/>
          </xsl:call-template>
          <xsl:text> TO </xsl:text>
          <xsl:call-template name="QuotedName">
           <xsl:with-param name="NameNode" select="sxml:NAME"/>
          </xsl:call-template>
         </xsl:element>
        </xsl:element>
       </xsl:if>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="GetConstraint">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="ColListItems" select="''"/>
  <xsl:param name="ConsType" select="''"/>
  <xsl:param name="ForModify">FALSE</xsl:param>
  <!-- ***************************************************************
Template: GetConstraint
Current Node: CONSTRAINT_LIST_ITEM
Parameters:
 ParentNode  - CONSTRAINT_LIST_ITEM
 ColListItems - COL_LIST_ITEM nodes
       ConsType - PRIMARY/FOREIGN/UNIQUE/CHECK/SCOPE
 ForModify - TRUE (constraint is being modified) 
                     FALSE (drop/create constraint)
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$ConsType='PRIMARY'">
    <xsl:text> PRIMARY KEY </xsl:text>
    <xsl:if test="$ForModify='FALSE'">
     <xsl:text>(</xsl:text>
     <xsl:call-template name="AlterCOL_LIST">
      <xsl:with-param name="ColListItems" select="$ColListItems"/>
     </xsl:call-template>
     <xsl:text>)</xsl:text>
    </xsl:if>
   </xsl:when>
   <xsl:when test="$ConsType='FOREIGN'">
    <xsl:text> FOREIGN KEY (</xsl:text>
    <xsl:call-template name="AlterCOL_LIST">
     <xsl:with-param name="ColListItems" select="$ColListItems"/>
    </xsl:call-template>
    <xsl:text>)</xsl:text>
    <!-- reference_clause -->
    <xsl:if test="sxml:REFERENCES">
     <xsl:text> REFERENCES "</xsl:text>
     <xsl:if test="$EMIT_SCHEMA=1">
      <xsl:value-of select="sxml:REFERENCES/sxml:SCHEMA"/>
      <xsl:text>"."</xsl:text>
     </xsl:if>
     <xsl:value-of select="sxml:REFERENCES/sxml:NAME"/>
     <xsl:text>"</xsl:text>
     <!-- using it can not get right result, don't know why 
     <xsl:call-template name="SchemaName">
      <xsl:with-param name="ParentNode" select="sxml:REFERENCES"/>
     </xsl:call-template>
-->
     <xsl:text>(</xsl:text>
     <xsl:call-template name="AlterCOL_LIST">
      <xsl:with-param name="ColListItems" select="sxml:REFERENCES/sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
     </xsl:call-template>
     <xsl:text>)</xsl:text>
     <xsl:if test="sxml:REFERENCES/sxml:ON_DELETE">
      <xsl:text> ON DELETE </xsl:text>
      <xsl:choose>
       <xsl:when test="sxml:REFERENCES/sxml:ON_DELETE='SET_NULL'">SET NULL</xsl:when>
       <xsl:otherwise>
        <xsl:value-of select="sxml:REFERENCES/sxml:ON_DELETE"/>
       </xsl:otherwise>
      </xsl:choose>
     </xsl:if>
    </xsl:if>
   </xsl:when>
   <xsl:when test="$ConsType='UNIQUE'">
    <xsl:text> UNIQUE (</xsl:text>
    <xsl:call-template name="AlterCOL_LIST">
     <xsl:with-param name="ColListItems" select="$ColListItems"/>
    </xsl:call-template>
    <xsl:text>)</xsl:text>
   </xsl:when>
   <xsl:when test="$ConsType='CHECK'">
    <xsl:text> CHECK (</xsl:text>
    <xsl:value-of select="sxml:CONDITION"/>
    <xsl:text>)</xsl:text>
   </xsl:when>
   <xsl:when test="$ConsType='SCOPE'">
    <xsl:text> SCOPE FOR (</xsl:text>
    <xsl:call-template name="QuotedName">
     <xsl:with-param name="NameNode" select="sxml:COL"/>
    </xsl:call-template>
    <xsl:text>) IS </xsl:text>
    <xsl:call-template name="SchemaName">
     <xsl:with-param name="ParentNode" select="sxml:REFERENCES"/>
    </xsl:call-template>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="AlterConstraintState">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="TypeNode" select="''"/>
  <!-- *******************************************************************
Template: AlterConstraintState
Current node: parent of ENABLE, DISABLE, etc.
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 TypeNode   - ADD or '' (modify)
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$TypeNode='ADD'">
    <!-- add constraint_state for add constraint ddl  -->
    <xsl:if test="$ParentNode/sxml:INITIALLY_DEFERRED/@src='2' or $ParentNode/sxml:INITIALLY_DEFERRED/@src!='1'"> INITIALLY DEFERRED</xsl:if>
    <!-- In the ADD case we only need to see if we need to propagate the RELY (i.e., should never have a src or value1 -->
    <xsl:if test="$ParentNode/sxml:RELY"> RELY</xsl:if>
    <!-- [NO]VALIDATE -->
    <xsl:if test="$ParentNode/sxml:USING_INDEX">
     <!-- need more , partitioning -->
     <xsl:text> USING INDEX</xsl:text>
     <xsl:call-template name="AlterPhysicalAttributes">
      <xsl:with-param name="ParentNode" select="$ParentNode/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES"/>
      <xsl:with-param name="TypeNode" select="$TypeNode"/>
     </xsl:call-template>
     <!-- hui -->
     <xsl:choose>
      <xsl:when test="$ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING">
       <xsl:call-template name="GlobalPartIndex">
        <xsl:with-param name="ParentNode" select="$ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING"/>
        <xsl:with-param name="TypeNode" select="$TypeNode"/>
       </xsl:call-template>
      </xsl:when>
      <xsl:when test="$ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING">
       <xsl:call-template name="LocalPartIndex">
        <xsl:with-param name="ParentNode" select="$ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING"/>
        <xsl:with-param name="TypeNode" select="$TypeNode"/>
       </xsl:call-template>
      </xsl:when>
     </xsl:choose>
    </xsl:if>
    <!-- ENABLE/DISABLE -->
    <xsl:call-template name="ConstraintState2">
     <xsl:with-param name="ParentNode" select="."/>
    </xsl:call-template>
   </xsl:when>
   <!-- modify constraint_state  -->
   <xsl:otherwise>
    <xsl:if test="$ParentNode/sxml:INITIALLY_DEFERRED/@src='2'"> INITIALLY DEFERRED</xsl:if>
    <xsl:if test="$ParentNode/sxml:RELY/@src='2' or $ParentNode/sxml:RELY/@src!='1'"> RELY</xsl:if>
    <xsl:if test="$ParentNode/sxml:RELY/@src='1' or $ParentNode/sxml:RELY/@src!='2'"> NORELY</xsl:if>
    <xsl:if test="$ParentNode/sxml:USING_INDEX/@src='2' or
       $ParentNode/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:PCTFREE/@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:INITRANS/@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:STORAGE/sxml:NEXT/@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES/sxml:STORAGE/sxml:MINEXTENTS/@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING/@src or
       $ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM//@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM/sxml:SEGMENT_ATTRIBUTES//@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING/@src or
       $ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM//@value1 or
       $ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM/sxml:SEGMENT_ATTRIBUTES//@value1">
     <!-- need more , partitioning -->
     <xsl:text> USING INDEX</xsl:text>
     <xsl:call-template name="AlterPhysicalAttributes">
      <xsl:with-param name="ParentNode" select="$ParentNode/sxml:USING_INDEX/sxml:INDEX_ATTRIBUTES"/>
      <xsl:with-param name="TypeNode">
       <xsl:if test="$ParentNode/sxml:USING_INDEX/@src='2'">ADD</xsl:if>
      </xsl:with-param>
     </xsl:call-template>
     <xsl:choose>
      <xsl:when test="$ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING">
       <xsl:call-template name="GlobalPartIndex">
        <xsl:with-param name="ParentNode" select="$ParentNode/sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING"/>
        <xsl:with-param name="TypeNode" select="$TypeNode"/>
       </xsl:call-template>
      </xsl:when>
      <xsl:when test="$ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING">
       <xsl:call-template name="LocalPartIndex">
        <xsl:with-param name="ParentNode" select="$ParentNode/sxml:USING_INDEX/sxml:LOCAL_PARTITIONING"/>
        <xsl:with-param name="TypeNode" select="$TypeNode"/>
       </xsl:call-template>
      </xsl:when>
     </xsl:choose>
    </xsl:if>
    <!-- ENABLE/DISABLE -->
    <xsl:call-template name="ConstraintState2">
     <xsl:with-param name="ParentNode" select="."/>
    </xsl:call-template>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="GlobalPartIndex">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="TypeNode" select="''"/>
  <!-- *******************************************************************
Template: GlobalPartIndex
Current node: sxml:USING_INDEX/sxml:GLOBAL_PARTITIONING
Parameters:
 ParentNode - GLOBAL_PARTITIONING
 TypeNode   - ADD
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$ParentNode/sxml:HASH_PARTITIONING">
    <xsl:text> GLOBAL PARTITION BY HASH (</xsl:text>
    <xsl:call-template name="AlterCOL_LIST">
     <xsl:with-param name="ColListItems" select="$ParentNode/sxml:HASH_PARTITIONING/sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
    </xsl:call-template>
    <xsl:text>) </xsl:text>
    <xsl:text>(</xsl:text>
    <xsl:call-template name="IndexPartList">
     <xsl:with-param name="PartList" select="$ParentNode/sxml:HASH_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM"/>
    </xsl:call-template>
    <xsl:text>) </xsl:text>
   </xsl:when>
   <xsl:when test="$ParentNode/sxml:RANGE_PARTITIONING">
    <xsl:text> GLOBAL PARTITION BY RANGE (</xsl:text>
    <xsl:call-template name="AlterCOL_LIST">
     <xsl:with-param name="ColListItems" select="$ParentNode/sxml:RANGE_PARTITIONING/sxml:COL_LIST/sxml:COL_LIST_ITEM"/>
    </xsl:call-template>
    <xsl:text> )</xsl:text>
    <xsl:text>(</xsl:text>
    <xsl:call-template name="IndexPartList">
     <xsl:with-param name="PartList" select="$ParentNode/sxml:RANGE_PARTITIONING/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM"/>
    </xsl:call-template>
    <xsl:text>) </xsl:text>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="LocalPartIndex">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="TypeNode" select="''"/>
  <!-- *******************************************************************
Template: LocalPartIndex
Current node: sxml:USING_INDEX/sxml:LOCAL_PARTITIONING
Parameters:
 ParentNode - LOCAL_PARTITIONING
 TypeNode   - ADD
******************************************************************** -->
  <xsl:text> LOCAL (</xsl:text>
  <xsl:call-template name="IndexPartList">
   <xsl:with-param name="PartList" select="$ParentNode/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM"/>
  </xsl:call-template>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template name="IndexPartList">
  <xsl:param name="PartList" select="''"/>
  <!-- *******************************************************************
Template: IndexPartList
Parameters:
 IndexPartList - PARTITION_LIST 

******************************************************************** -->
  <xsl:for-each select="$PartList">
   <xsl:if test="position()!=1">
    <xsl:text>,</xsl:text>
   </xsl:if>
   <xsl:text>PARTITION "</xsl:text>
   <xsl:value-of select="sxml:NAME"/>
   <xsl:text>" </xsl:text>
   <xsl:if test="sxml:VALUES">
    <xsl:text>VALUES LESS THAN (</xsl:text>
    <xsl:value-of select="sxml:VALUES"/>
    <xsl:text>) </xsl:text>
   </xsl:if>
   <xsl:if test="sxml:SEGMENT_ATTRIBUTES">
    <xsl:call-template name="AlterSegmentAttributes">
     <xsl:with-param name="ParentNode" select="sxml:SEGMENT_ATTRIBUTES"/>
     <xsl:with-param name="TypeNode" select="'ADD'"/>
    </xsl:call-template>
   </xsl:if>
   <xsl:if test="sxml:SUBPARTITION_LIST">
    <xsl:text>( </xsl:text>
    <xsl:for-each select="sxml:SUBPARTITION_LIST/sxml:SUBPARTITION_LIST_ITEM">
     <xsl:if test="position()!=1">
      <xsl:text>,</xsl:text>
     </xsl:if>
     <xsl:text>SUBPARTITION "</xsl:text>
     <xsl:value-of select="sxml:NAME"/>
     <xsl:text>" </xsl:text>
     <xsl:if test="sxml:TABLESPACE">
      <xsl:text>TABLESPACE </xsl:text>
      <xsl:value-of select="sxml:TABLESPACE"/>
     </xsl:if>
    </xsl:for-each>
    <xsl:text>)</xsl:text>
   </xsl:if>
   <xsl:if test="sxml:TABLESPACE">
    <xsl:text> TABLESPACE "</xsl:text>
    <xsl:value-of select="sxml:TABLESPACE"/>
    <xsl:text>"</xsl:text>
   </xsl:if>
  </xsl:for-each>
 </xsl:template>
</xsl:stylesheet>
